(ns ventas-devtools.repl
  "REPL-driven development.
   Don't add ventas.* aliases in this ns, as it conflicts with dynamic aliases."
  (:require
   [cemerick.pomegranate :as pomegranate]
   [cemerick.pomegranate.aether :as aether]
   [figwheel-sidecar.repl-api :as figwheel]
   [clojure.tools.namespace.repl :as tn]
   [mount.core :as mount]
   [ventas.config]
   [ventas.events]
   [ventas.theme]
   [ventas-devtools.client :as client]
   ;; Aliases for REPL usage
   [clojure.repl :refer :all]
   [clojure.core.async :as core.async :refer [<! chan >! go]]
   [clojure.spec.alpha :as spec]
   [taoensso.timbre :as timbre]))

;; Let Clojure warn you when it needs to reflect on types.
;; Type annotations should be added in such cases to prevent degraded performance.
(alter-var-root #'*warn-on-reflection* (constantly true))

(cond->> ["src/clj" "src/cljc" "test/clj" "test/cljc"]
         (not (ventas.config/get :strict-classloading)) (concat ["dev"])
         true (apply clojure.tools.namespace.repl/set-refresh-dirs))

(def cljs-repl figwheel/cljs-repl)

;; Some Emacs users might need to avoid the "dynamic classloading" that ventas
;; does when on development. Basically, if something fails to start, dynamic
;; classloading lets you correct it while having a functional REPL.
;; Doing this require when loading this namespace will force you to restart
;; the REPL process in order to fix the error.
;; As said, use it if you need it.
(when (ventas.config/get :strict-classloading)
  (require 'ventas.core))

(def aliases
  {'api 'ventas.server.api
   'common.utils 'ventas.common.utils
   'config 'ventas.config
   'd 'datomic.api
   'db 'ventas.database
   'entity 'ventas.database.entity
   'plugin 'ventas.plugin
   'schema 'ventas.database.schema
   'search 'ventas.search
   'seed 'ventas.database.seed
   'server 'ventas.server
   'site 'ventas.site
   'utils 'ventas.utils
   'ws 'ventas.server.ws})

(defn deinit-aliases []
  (doseq [[from to] aliases]
    (ns-unalias 'ventas-devtools.repl from)))

(defn init-aliases []
  (deinit-aliases)
  (doseq [[from to] aliases]
    (alias from to)))

(defn add-dependency [coordinates]
  (pomegranate/add-dependencies
   :coordinates [coordinates]
   :repositories (merge aether/maven-central
                        {"clojars" "https://clojars.org/repo"})))

;; Lifecycle
(defn init []
  (deinit-aliases)
  (mount/start)
  (init-aliases)
  (core.async/put! (ventas.events/pub :init) true)
  :done)

(defn reset []
  (deinit-aliases)
  (mount/stop)
  (tn/refresh :after 'mount/start)
  (init-aliases)
  (core.async/put! (ventas.events/pub :init) true)
  :done)

(defn keyword->states [kw]
  (get {:figwheel '[client/figwheel]
        :sass '[client/sass]
        :db '[ventas.database/conn]
        :indexer '[ventas.search/indexer]
        :server '[ventas.server/server]
        :config '[ventas.config/config-loader]
        :kafka '[ventas.kafka.registry/registry
                 ventas.kafka.producer/producer
                 ventas.stats.indexer/indexer]}
       kw))

(defn r
  "Reloads changed namespaces, and restarts the defstates within them.
   Accepts optional keywords representing defstates to restart (regardless of
   a need for it)
   Example:
     (r :figwheel :db)
   Refer to keyword->state to see what states can be restarted in this way"
  [& states]
  (when (= (ns-name *ns*) 'ventas-devtools.repl)
    (deinit-aliases))
  (let [states (->> states
                    (mapcat (fn [kw]
                              (let [states (keyword->states kw)]
                                (when-not states
                                  (throw (Exception. (str "State " kw " does not exist"))))
                                states)))
                    (map (fn [state]
                           (ns-resolve 'ventas-devtools.repl state))))
        _ (when (seq states)
            (apply mount/stop states))
        result (tn/refresh)]
    (when (instance? Throwable result)
      (throw result))
    (when (seq states)
      (apply mount/start states))
    (when (= (ns-name *ns*) 'ventas-devtools.repl)
      (init-aliases))
    (core.async/put! (ventas.events/pub :init) true)
    :done))

(defn run-tests []
  (clojure.test/run-all-tests #"ventas.*?\-test"))

(defn set-theme! [theme]
  (reset! client/figwheel-theme theme)
  (mount.core/stop (ns-resolve 'ventas-devtools.repl 'client/figwheel))
  (mount.core/start (ns-resolve 'ventas-devtools.repl 'client/figwheel)))
