(ns org.bituf.sqlrat.entity
  (:use org.bituf.sqlrat.entity.internal)
  (:use [clojure.contrib.sql :as sql :only ()]))


(def *show-sql* true)
(def *show-sql-results* false)


; ===== Utility functions and macros =====

(defmacro in-db [db & body]
  `(sql/with-connection ~db
    ~@body))


(defmacro in-txn [db & body]
  `(sql/with-connection ~db
    (sql/transaction
      ~@body)))


(defn db-query [query-vec]
  "Fetch rows from database table. Execute this with in-db or in-txn."
  (if *show-sql* (mypp "Executing SQL..." query-vec))
  (let [result (sql/with-query-results rows
                 query-vec
                 (if (nil? rows) nil (into [] rows)))]
    (if *show-sql-results* (mypp "SQL Result..." result))
    result))


;(defn save-row [^Keyword table ^Map row ^Keyword id-column]
;  "Save given row. Table should be specified as :tablename (keyword).
;  Row is simply a map of :columnName to value. Execute with in-txn or in-db."
;  (let [id=? (str (name id-column) "=?")]
;    (sql/update-or-insert-values
;        table
;        [id=? (id-column row)]
;        row)))

(defn save-row [^Keyword table ^Map row ^Keyword id-column]
  "Save given row. Table should be specified as :tablename (keyword).
  Row is simply a map of :columnName to value. Execute with in-txn or in-db.
  Returns the saved row, which may have generated ID (if applicable)."
  (let [id=? (str (name id-column) "=?")]
    (let [result (update-or-insert-values-returnid
                   table [id=? (id-column row)] row)]
      (let [generated-key (:generated_key (first result))]
        (if (nil? generated-key) row
          (assoc row id-column generated-key))))))


;;; ===== Entity relationships =====

;; relation of this table with another table
(defrecord RelationMetadata
  [this-column   ; (keyword) column in this entity
   that-entity   ; EntityMetadata instance for the other entity
   that-column   ; (keyword) column name in that entity
   that-depends? ; whether the other entity depends on "this"
   ] )


(defprotocol Relation
  (rel-meta [this] "Return a sequence of Relation objects"))


(defn relation
  ([^Keyword this-col ^EntityMetadata that-ent ^Keyword that-col that-depends?]
    (RelationMetadata. this-col that-ent that-col that-depends?))
  ([^Keyword this-col ^EntityMetadata that-ent ^Keyword that-col]
    (RelationMetadata. this-col that-ent that-col false)))


(defn one-to-many [^Keyword this-col ^EntityMetadata that-ent-meta ^Keyword that-col]
  (relation this-col that-ent-meta that-col true))


(defn many-to-one [^Keyword this-col ^EntityMetadata that-ent-meta ^Keyword that-col]
  (relation this-col that-ent-meta that-col false))


(def one-to-one-depends one-to-many)


(def one-to-one         many-to-one)


(defn rel-impl [rels-vector]
  "Returns implementation for the Relation protocol."
  {:rel-meta (fn [this] (as-vector rels-vector))} )


;;; ===== Entity definition =====

(defn to-row [entity]
  "Default implementation for to-row-fn."
  (into {} entity))


(defmacro from-row [& entity-creator]
  `#(~@entity-creator {} %))


(defrecord EntityMetadata
  [name ;; :entry (keyword) name of the entity
   id   ;; :autoid (keyword) name of the ID column
   from-row-fn] ;; factory fn: IN row, OUT entity
                ;; (from-row Entity.)
   ;;;
   ;; ##### Optional items with examples #####
   ;;
   ;; ===== columns definition (required for create-table):
   ;;
   ;; :cols  [[:autoid     :int "NOT NULL PRIMARY KEY AUTO_INCREMENT"]
   ;;         [:entryid    :int           "NOT NULL"]
   ;;         [:content    "varchar(500)" "NOT NULL"]
   ;;         [:whenposted "DATETIME"     "NOT NULL"]
   ;;         [:isdeleted  "BOOLEAN"      "NOT NULL DEFAULT false"]
   ;;         [:name       "varchar(30)"  "NOT NULL"]
   ;;         [:email      "varchar(50)"  "NOT NULL"]
   ;;         [:url        "varchar(100)"]]
   ;;
   ;; ===== to-row function to convert from entity to row
   ;;       (default implementation is used if not specified)
   ;;
   ;; :to-row-fn  to-row
   )


(defn entity-meta [name id from-row-fn
                   & {:keys [cols to-row-fn] :or {to-row-fn to-row}}]
  "Factory function to create entity metadata."
  (EntityMetadata. name id from-row-fn
    {} {:cols cols :to-row-fn to-row-fn}))


(defprotocol Entity ;; represents a database table row
  (get-meta [this] "Get entity metadata"))


(defn entity-impl [^EntityMetadata e-meta]
  "Returns implementation for Entity protocol."
  {:get-meta (fn [this] e-meta)} )


(defn extend-entity
  ([record ent-meta]
    (extend record
      Entity (entity-impl ent-meta)))
  ([record ent-meta rel-metadata]
    (extend record
      Entity   (entity-impl ent-meta)
      Relation (rel-impl rel-metadata))))


(def count-col "COUNT(*) AS cnt")


(defn read-count-col [row-or-entity]
  (:cnt (first row-or-entity)))


(defn get-id-column [entity]
  "Return ID column from entity"
  (:id (get-meta entity)))


(defn get-id-value [entity]
  "Return ID column value from entity"
  ((get-id-column entity) entity))


;;; ===== Functions to work on entity and entity metadata.
;;;       Execute these with in-db / in-txn

;; function that accepts the (rel-meta entity) and returns a map
;; {:that-entity-name each-rel}
(def dbrel-lookup-by-that-entity
  (memoize
    (fn [rels-vector]
      (let [rel-vector (as-vector rels-vector)
            that-map (transient {})]
        (doseq [each rel-vector]
          (assoc! that-map (:name (:that-entity each)) each))
        (persistent! that-map)))))


(defn create-table [^EntityMetadata entity-meta]
  "Create the table."
  (let [table-name (:name entity-meta)
        cols-spec  (:cols entity-meta)]
    (apply sql/create-table table-name cols-spec)))


(defn drop-table [^EntityMetadata entity-meta]
  "Drop the table."
  (sql/drop-table (:name entity-meta)))


(defn find-by-sql [^EntityMetadata entity-meta sql-vec]
  "Find entities with custom SQL/criteria. This is a free-form style"
  (let [sql-vector (as-vector sql-vec)
        rows       (db-query sql-vector)]
    (into [] (for [each rows] ((:from-row-fn entity-meta) each)))))


(defn find-by-criteria [^EntityMetadata entity-meta &
                        {:keys [cols where] :or {cols ["*"] where []}}]
  "Find entities using :cols and/or :where attributes. Examples are below:
   :cols [:title :content \"whenposted\"]
   :where [\"whenposted < ? ORDER BY whenposted\" (new java.util.Date)]
   "
  (let [no-cols?     (or (nil? cols) (empty? cols))
        no-where?    (or (nil? where) (empty? where))
        cols-vector  (if no-cols? nil (as-vector cols))
        cols-str     (if no-cols? "*"
                       (.replace
                         (apply str (interpose ", " cols-vector)) ":" ""))
        from-str     (name (:name entity-meta))
        where-vector (if no-where? nil (as-vector where))
        where-str    (if no-where? "" (str " WHERE " (first where-vector)))
        where-arg    (if no-where? [] (rest where-vector))
        sql-vector   (into
                       [(str "SELECT " cols-str " FROM " from-str where-str)]
                       where-arg)]
    (find-by-sql entity-meta sql-vector)))


(defn find-by-id [^EntityMetadata entity-meta id &
                  {:keys [cols where] :or {cols ["*"] where []}}]
  "Find an entity of given type for a given ID. You can also pass :cols and
   :where attributes as in find-by-criteria."
  (let [no-where?    (or (nil? where) (empty? where))
        where-vector (if no-where? nil (as-vector where))
        where-str    (if no-where? "" (str " AND " (first where-vector)))
        where-arg    (if no-where? [] (rest where-vector))
        rows         (find-by-criteria entity-meta
                       :cols  cols
                       :where (into
                                [(str (name (:id entity-meta)) "=?" where-str)
                                 id]
                                where-arg))]
    (if (empty? rows)
      nil
      ((:from-row-fn entity-meta) (first rows)))))


(defn save [^Entity entity]
  "Save given entity"
  (let [ent-meta (get-meta entity)
        from-row-fn (:from-row-fn ent-meta)]
    (from-row-fn
      (save-row
        (:name ent-meta) ((:to-row-fn ent-meta) entity) (:id ent-meta)))))


(defn delete
  ([entity-meta id]
    "Delete by ID"
    (sql/delete-rows (:name entity-meta)
      [(str (name (:id entity-meta)) "=?") id]))
  ([entity]
    "Delete given entity"
    (delete (get-meta entity) (get-id-value entity))))


;;; ===== Relationship handling functions. Execute with in-db / in-txn

(defn find-rels [entities-vec ^EntityMetadata that-meta &
                 {:keys [cols where] :or {cols ["*"] where []}}]
  "Fetch related entities. You can use the :cols and :where attributes as in
   find-by-criteria function. This avoids N+1 Selects. Returns a map in the form
   
   {entity1 [e1-rel1 e1-rel2 ...]
    entity2 [e2-rel1 e2-rel2 e2-rel3 ...]}
   
   Entities with no children are not included in the map."
  (let [entities (into #{} (as-vector (if (map? entities-vec) [entities-vec]
                                        entities-vec)))]
    ;; error check
    (if (or (nil? entities)  (empty? entities) (nil? (first entities))
          (let [entity-meta  (get-meta (first entities))
                invalid?    #(or (nil? %)
                               (not= entity-meta (get-meta %)))]
            (some invalid? entities)))
      (throw (IllegalArgumentException.
               "One or more non-null entities of same type expected")))
    ;; actual processing
    (let [entity         (first entities)
          no-where?      (or (nil? where) (empty? where))
          where-vector   (if no-where? nil (as-vector where))
          where-str      (if no-where? "" (str " AND " (first where-vector)))
          where-arg      (if no-where? [] (rest where-vector))
          this-meta      (get-meta entity)
          that-table-map (dbrel-lookup-by-that-entity (rel-meta entity))
          rel-data       (that-table-map (:name that-meta))
          that-column    (:that-column rel-data)
          this-column    (:this-column rel-data)
          rel-col-values (map #(this-column %) entities)
          ??-placeholder (apply str
                           (interpose ", "
                             (take (count entities) (repeat \?))))
          add-rel-column (fn [few-cols]
                           (if (some #(or (= that-column %) (= "*" %)) few-cols)
                             few-cols
                             (conj few-cols that-column)))
          cols-vector    (add-rel-column (as-vector cols))
          group-by-str   (if (and (some #(= count-col %) cols-vector)
                               (< 1 (count entities)))
                           (str " GROUP BY " (name that-column)))
          ;; fetch relations
          child-entities (find-by-criteria that-meta
                           :cols  cols-vector
                           :where (into (into
                                          [(str (name that-column)
                                             " IN (" ??-placeholder ")"
                                             where-str group-by-str)]
                                          rel-col-values)
                                    where-arg))
          find-parent    (fn [child-entity]
                           (let [foreign-key (that-column child-entity)]
                             (first (filter #(= (this-column %) foreign-key)
                                      entities))))]
      (group-by find-parent child-entities))))


(defn save-deps [^Entity entity deps-vector]
  "Save dependents (THAT table) -- 1-to-many (has-many) relationships"
  (let [cvec (as-vector deps-vector)
        rels (rel-meta entity)
        that-table-map (dbrel-lookup-by-that-entity rels)]
    (into [] (for [each cvec]
      (if-let [each-rel (that-table-map (:name (get-meta each)))]
        (let [child (assoc each
                      (:that-column each-rel)
                      ((:this-column each-rel) entity))]
          (save child)))))))


(defn find-siblings [^Entity entity ^Entity rel-entity &
                     {:keys [cols where] :or {cols ["*"] where []}}]
  "Fetch sibling entities - Many-to-1 relationships. You can use the :cols and
   :where attributes as in find-by-criteria function."
  (let [no-where?       (or (nil? where) (empty? where))
        where-vector    (if no-where? nil (as-vector where))
        where-str       (if no-where? "" (str " AND " (first where-vector)))
        where-arg       (if no-where? [] (rest where-vector))
        this-meta       (get-meta entity)
        that-table-map  (dbrel-lookup-by-that-entity (rel-meta entity))
        rel-data        (that-table-map (:name (get-meta rel-entity)))
        this-table-name (name (:name this-meta))
        this-col-name   (name (:this-column rel-data))
        that-id-value   ((:that-column rel-data) rel-entity)]
    (find-by-criteria this-meta
      :cols  cols
      :where (into [(str this-col-name "=?" where-str) that-id-value]
                 where-arg))))


(defn delete-cascade [entity]
  [entity]
  "Delete (cascaded) a given entity"
  (let [rels (rel-meta entity)]
    (doseq [each rels]
      (if (:that-depends? each)
        (let [c ((find-rels entity (:that-entity each)) entity)]
          (doseq [each-child c]
            (delete-cascade each-child))))))
  (delete entity))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; pretty-printing for rows

(def max-col-print-width 40)
(def delim " | ")

(defn print-entities [entities]
  "Print homogenous entities in a table format. Keys from the first entity are
   used as title. Passing an empty sequence of entities prints nothing at all."
  (if-let [rows (map to-row (as-vector (if (map? entities) [entities] entities)))]
    (let [cols-count (count (first rows))
          cols-width (atom (into [] (take cols-count (repeat 0))))
          keys-as-str (map name (keys (first rows)))
          keys-n-vals (conj (map vals rows) keys-as-str)
          ;; translate non-printable chars http://hyperpolyglot.wikidot.com/lisp
          xlate-np-chars (fn [fs]
                           (let [xl {"\b" "\\b" "\f" "\\f" "\n" "\\n"
                                     "\r" "\\r" "\t" "\\t"}
                                 ks (keys xl)]
                             (apply str
                               (map #(let [s (str %)]
                                       (if (.contains ks s) (get xl s) s))
                                 fs))))
          ;(fn [s] (.replace (.replace s "\n" "\\n") "\t" "\\t"))
          ]
      ;; pass #1 -- calculate width of columns
      (doseq [each keys-n-vals]
        (let [each-cols-width (map #(count (xlate-np-chars (str %))) each)
              max-cols-width (map max each-cols-width @cols-width)]
          ;; keep the maximum col width under limits
          (reset! cols-width
            (map min
              max-cols-width (take cols-count (repeat max-col-print-width))))))
      ;; pass #2 -- actually print the cols
      (let [fixed-width-str (fn [text width]
                              (let [padded-text (apply str (xlate-np-chars text)
                                                  (take width (repeat \ )))]
                                (apply str (take width padded-text))))
            print-cols (fn [cols]
                         (println
                           (apply str
                             (interpose delim
                               (map fixed-width-str cols @cols-width)))))]
        ;; print titles and rows
        (print-cols keys-as-str) ;; column titles
        (print-cols (map #(apply str (repeat % "-")) @cols-width)) ;; dashes
        (doseq [each-row rows] ;; column values
          (print-cols (map str (vals each-row))))))))
