(ns net.lewisship.cli-tools
  "Utilities for create CLIs around functions, and creating tools with multiple sub-commands."
  (:require [net.lewisship.cli-tools.impl :as impl]
            [clojure.string :as str]))

(defn exit
  "An indirect call to System/exit, passing a numeric status code (0 for success, non-zero for
  an error).

  This is provided so that, during testing, when [[set-prevent-exit!]] has been called, the call
  to `exit` will instead throw an exception."
  [status]
  (impl/exit status))

(defn set-prevent-exit!
  "Normally, after displaying a command summary, `System/exit` is called (with 0 if for --help,
   or 1 if a validation error).

   For testing purposes, this can be prevented; instead, an exception is thrown,
   with message \"Exit\" and ex-data {:status <status>}."
  [flag]
  (alter-var-root #'impl/prevent-exit (constantly flag)))

(defn print-summary
  "Prints the command's summary to `*out*`; partially generated by clojure.tools.cli, and then
  enhanced with more information about positional command line arguments.

  This is often used when a command performs additional validation of its arguments
  and needs to output the summary and errors on failure.

  Uses the command map that is available in `defcommand` function
  (using the :as clause).

  errors is a seq of strings to display as errors."
  [command-map errors]
  (impl/print-summary command-map errors))

(defn best-match
  "Given an input string and a seq of possible values, returns the matching value if it can
  be uniquely identified.

  Values may be strings, symbols, or keywords.

  best-match does a caseless substring match against the provided values. It returns the single
  value that matches the input. It returns nil if no value matches, or if multiple values match.

  Some special handling for the `-` character; the input value is split on `-` and turned into
  a generous regular expression that matches the substring on either side of the `-` as well as the `-`
  itself.

  Returns the string/symbol/keyword from values.

  e.g. `:parse-fn #(cli-tools/best-match % #{:red :green :blue})` would parse an input of `red` to
  `:red`, or an input of `b` to `:blue`; `z` matches nothing and returns nil, as would
  `e` which matches multiple values.

  Expects symbols and keywords to be unqualified."
  [input values]
  (let [m (reduce (fn [m v]
                    (assoc m (name v) v))
                  {}
                  values)
        matches (impl/find-matches input (keys m))]
    (when (= 1 (count matches))
      (get m (first matches)))))

(defn sorted-name-list
  "Converts a seq of strings, keywords, or symbols (as used with [[best-match]]) to a comma-separated
  string listing the values. This is often used with help summary or error messages related."
  [values]
  (->> values
       (map name)
       sort
       (str/join ", ")))

(defmacro defcommand
  "Defines a command.

   A command's _interface_ identifies how to parse command options and positional arguments,
   mapping them to local symbols.

   Commands must always have a docstring; this is part of the `-h` / `--help` summary.

   The returned function is variadic, accepting a number of strings, much
   like a `-main` function. For testing purposes, it may instead be passed a single map,
   a command map, which bypasses parsing and validation of the arguments, and is used only for testing.

   Finally, the body is evaluated inside a let that destructures the options and positional arguments into local symbols."
  [command-name docstring interface & body]
  (assert (simple-symbol? command-name)
          "defcommand expects a symbol for command name")
  (assert (string? docstring)
          (throw "defcommand requires a docstring"))
  (assert (vector? interface)
          "defcommand expects a vector to define the interface")
  (let [symbol-meta (meta command-name)
        parsed-interface (impl/compile-interface docstring interface)
        {:keys [option-symbols command-map-symbol command-summary let-forms validate-cases]
         :or {command-map-symbol (gensym "command-map-")}} parsed-interface
        command-name' (or (:command-name parsed-interface)
                          (name command-name))
        let-option-symbols (cond-> []
                             (seq option-symbols)
                             (into `[{:keys ~option-symbols} (:options ~command-map-symbol)]))
        symbol-with-meta (cond-> (assoc symbol-meta
                                        :doc docstring
                                        :arglists '[['& 'args]]
                                        ::impl/command-name command-name')
                           command-summary (assoc ::impl/command-summary command-summary))
        ;; Keys actually used by parse-cli and print-summary
        parse-cli-keys [:command-args :command-options :parse-opts-options :command-doc :summary]
        validations (when (seq validate-cases)
                      `(when-let [message# (cond ~@(impl/invert-tests-in-validate-cases validate-cases))]
                         (print-summary ~command-map-symbol [message#])
                         (exit 1)))]
    `(defn ~command-name
       ~symbol-with-meta
       [~'& args#]
       (let [~@let-forms
             ;; args# is normally a seq of strings, from *command-line-arguments*, but for testing,
             ;; it can also be a map with key :options
             test-mode?# (impl/command-map? args#)
             ~command-map-symbol (if test-mode?#
                                   (first args#)
                                   (impl/parse-cli ~command-name'
                                                   args#
                                                   ~(select-keys parsed-interface parse-cli-keys)))
             ~@let-option-symbols]
         (when-not test-mode?#
           ~validations)
         ~@body))))

(defcommand help
  "List available commands"
  []
  (impl/show-tool-help))

(defn- source-of
  [v]
  (str (-> v meta :ns ns-name) "/" (-> v meta :name)))

(defn- resolve-ns
  [ns-symbol]
  (if-let [ns-object (find-ns ns-symbol)]
    ns-object
    (throw (RuntimeException. (format "namespace %s not found (it may need to be required)" (name ns-symbol))))))

(defn locate-commands
  "Passed a seq of symbols identifying *loaded* namespaces, this function
  locates commands, functions defined by [[defcommand]].

  Normally, this is called from [[dispatch]] and is only needed when calling [[dispatch*]] directly.

  Returns a map from string command name to command Var."
  [namespace-symbols]
  (let [f (fn [m namespace-symbol]
            (->> (resolve-ns namespace-symbol)
                 ns-publics
                 vals
                 (reduce (fn [m v]
                           (let [command-name (-> v meta ::impl/command-name)]
                             (cond
                               (nil? command-name)
                               m

                               (contains? m command-name)
                               (throw (RuntimeException. (format "command %s defined by %s conflicts with %s"
                                                                 command-name
                                                                 (source-of v)
                                                                 (source-of (get m command-name)))))

                               :else
                               (assoc m command-name v))))
                         m)))]
    (reduce f {} namespace-symbols)))

(defn dispatch*
  "Invoked by [[dispatch]] after namespace and command resolution.

  This can be used, for example, to avoid including the builtin help command
  (or when providing an override).

  options:
  
  - :tool-name - used in command summary and errors
  - :tool-doc - used in command summary
  - :arguments - seq of strings; first is name of command, rest passed to command
  - :commands - map from string command name to a var (defined via [[defcommand]])

  All options are required.

  Returns nil."
  [options]
  (impl/dispatch options))

(defn dispatch
  "Locates commands in namespaces, finds the current command
  (as identified by the first command line argument) and processes CLI options and arguments.

  options:
  
  - :tool-name (optional, string) - used in command summary and errors
  - :tool-doc (optional, string) - used in help summary
  - :arguments - command line arguments to parse (defaults to `*command-line-args*`)
  - :namespaces - symbols identifying namespaces to search for commands

  The :tool-name option is only semi-optional; in a Babashka script, it will default
  from the `babashka.file` system property, if any. An exception is thrown if :tool-name
  is not provided and can't be defaulted.

  The default for :tool-doc is the docstring of the first namespace.

  dispatch will load any namespaces specified, then scan those namespaces to identify commands.
  It also adds a `help` command from this namespace.

  If option and argument parsing is unsuccessful, then
  a command usage summary is printed, along with errors, and the program exits
  with error code 1.

  dispatch simply loads and scans the namespaces, adds the `help` command, then calls [[dispatch*]].

  Returns nil."
  [options]
  (let [{:keys [namespaces arguments tool-name tool-doc]} options
        tool-name' (or tool-name
                       (impl/default-tool-name)
                       (throw (ex-info "No :tool-name specified" {:options options})))
        _ (when-not (seq namespaces)
            (throw (ex-info "No :namespaces specified" {:options options})))
        ;; Add this namespace, to include the help command by default
        commands (do
                   ;; Load all the other namespaces first
                   (run! require namespaces)
                   ;; Ensure built-in help command is first
                   (locate-commands (cons 'net.lewisship.cli-tools namespaces)))]
    (dispatch* {:tool-name tool-name'
                :tool-doc (or tool-doc
                              (some-> namespaces first find-ns meta :doc))
                :commands commands
                :arguments (or arguments *command-line-args*)})))
