(ns hub.queue.client
  (:require [clojure.core.async :as a]
            [com.stuartsierra.component :as c]
            [hub.queue.service :as q]
            [schema.core :as s]
            [taoensso.carmine :as car])
  (:import [com.stuartsierra.component Lifecycle]))
"Client for the RaceHub message bus."

(s/defn publish! :- s/Int
  "Publishes the given payload on the topic. Returns the number of subscribers."
  [topic :- s/Str
   payload :- s/Any]
  (if (not-empty topic)
    (try  (q/wcar (car/publish topic payload))
          (catch Exception e
            e))
    (throw (ex-info "Empty topic." {:type "empty_topic"}))))


(s/defn subscribe! :- (s/named (s/=> (s/pred nil?)) "unsubscribe fn")
  "Subscribes to the given topic. Drops messages for the topic onto
  the channel, as they come in. Returns a function of no args that can
  be called to unsubscribe from this topic."
  [topic :- s/Str
   channel]
  (if (not-empty topic)
    (try (let [listener (car/with-new-pubsub-listener (:spec @q/conf)
                          {topic (fn f1 [[msg-type _ msg]] (when (= msg-type "message")
                                                            (a/put! channel msg)))}
                          (car/subscribe  topic))]
           (fn []
             (car/with-open-listener listener (car/unsubscribe))
             (a/close! channel)))
         (catch Exception e
           e))
    (throw (ex-info "Empty topic." {:type "empty_topic"}))))

;; ## Service Startup
(s/defschema RedisSpec
  {(s/optional-key :host) s/Str
   (s/optional-key :port) s/Int
   (s/optional-key :password) s/Str
   (s/optional-key :timeout-ms) s/Int
   (s/optional-key :db) s/Int})

(s/defn queue-client :- Lifecycle
  "Client for the queue service."
  ([] (queue-client {}))
  ([spec :- RedisSpec]
   (reify c/Lifecycle
     (start [_] (reset! q/conf {:pool {} :spec spec}))
     (stop [_] (reset! q/conf nil)))))
