(ns hara.function.procedure.registry
  (:require [hara.core.base.check :as check]
            [hara.event :as event]
            [hara.data.base.map :as map]
            [hara.state :as state :refer [defcache]]))

(defcache +default-registry
  "default registry for running procedure instances"
  {:added "3.0"}
  [:atom {:tag "registry"
          :display (partial map/map-vals keys)}])

(defn add-instance
  "adds something to the registry
 
   @(add-instance (cache/cache {}) \"hello\" :1 {:a 1})
   => {\"hello\" {:1 {:a 1}}}"
  {:added "3.0"}
  ([name id instance]
   (add-instance +default-registry name id instance))
  ([registry name id instance]
   (state/update registry assoc-in [name id] instance)))

(defn remove-instance
  "removes something from the registry
 
   (-> (cache/cache {})
       (add-instance \"hello\" :1 {:a 1})
       (remove-instance \"hello\" :1)
       deref)
   => {}"
  {:added "3.0"}
  ([name id]
   (remove-instance +default-registry name id))
  ([registry name id]
   (state/update registry map/dissoc-in [name id])))

(defn list-instances
  "lists all items in the registry
 
   (-> (cache/cache {})
       (add-instance  \"hello\" :1 {:a 1})
       (add-instance  \"hello\" :2 {:b 1})
       (list-instances \"hello\"))
   => [{:a 1} {:b 1}]"
  {:added "3.0"}
  ([name] (list-instances +default-registry name))
  ([registry name]
   (vals (get @registry name))))

(defn get-instance
  "lists all items in the registry
 
   (-> (cache/cache {})
       (add-instance  \"hello\" :1 {:a 1})
       (add-instance \"hello\" :2 {:b 1})
       (get-instance \"hello\" :2))
   => {:b 1}"
  {:added "3.0"}
  ([name id] (get-instance +default-registry name id))
  ([registry name id]
   (get-in @registry [name id])))

(defn kill
  "kills the running instance in the registry
 
   (-> (cache/cache {})
       (add-instance \"hello\" :1 {:thread (volatile! (future (Thread/sleep 100000)))})
       (kill \"hello\" :1))
   => true"
  {:added "3.0"}
  ([name id] (kill +default-registry name id))
  ([registry name id]
   (when-let [{:keys [thread] :as instance} (get-instance registry name id)]
     (cond (future? @thread)
           (future-cancel @thread)
           
           (and (check/thread? @thread)
                (not= @thread (Thread/currentThread)))
           (do (.stop ^Thread @thread)
               (Thread/sleep 1)))
     (event/signal [:log {:msg "Killed Execution" :instance instance}])
     (remove-instance registry name id)
     true)))
