(ns hara.function.procedure
  (:require [hara.core.base.check :as check]
            [hara.data.base.map :as map]
            [hara.data.base.nested :as nested]
            [hara.function :as fn :refer [definvoke defexecutive]]
            [hara.function.procedure.middleware :as middleware]
            [hara.function.procedure.registry :as registry]
            [hara.function.procedure.retry :as retry]
            [hara.protocol.function :as protocol.function]
            [hara.state :refer [defcache]]
            [hara.time :as time]))

(defcache +default-cache
  "default cache for storing procedure results"
  {:added "3.0"}
  [:atom {:tag "cache"
          :display (fn [m]
                     (map/map-vals (fn [data]
                                     (map/map-vals keys data))
                                   m))}])

(defonce +default-settings+
  {:mode :async
   :interrupt false
   :time {:type java.util.Date
          :timezone (time/local-timezone)}
   :registry registry/+default-registry
   :cache    +default-cache
   :id-fn    (fn [_] (str (java.util.UUID/randomUUID)))})

(defn max-inputs
  "finds the maximum number of inputs that a function can take
 
   (max-inputs (fn ([a]) ([a b])) 4)
   => 2
 
   (max-inputs (fn [& more]) 4)
   => 4
 
   (max-inputs (fn ([a])) 0)
   => throws"
  {:added "3.0"}
  [func num]
  (if (fn/vargs? func)
    num
    (let [cargs (fn/arg-count func)
          carr (filter #(<= % num) cargs)]
      (if (empty? carr)
        (throw (Exception. (str "Function needs at least " (apply min cargs) " inputs")))
        (apply max carr)))))

(defrecord ProcedureInstance []
  Object
  (toString [obj]
    (str "#proc[" (:id obj) "]"
         (-> (select-keys obj [:name :result :timestamp :interrupt
                               :overwrite :cached :runtime
                               :mode :params :input :args :timeout :retry])
             (->> (into {}))
             (update-in [:runtime] (fn [x] (if x (deref x) {})))
             (update-in [:result]  (fn [x] (if (realized? x) (deref x) :waiting))))))

  clojure.lang.IDeref
  (deref [obj]
    (let [{:keys [data type]} (deref (:result obj))]
      (case type
        :success data
        :error   (throw data)))))

(defmethod print-method ProcedureInstance
  [v ^java.io.Writer w]
  (.write w (str v)))

(defn wrap-exception
  "creates a handler for retrying computation"
  {:added "3.0"}
  [f]
  (fn [{:keys [retry arglist] :as instance} args]
    (try (f instance args)
         (catch Throwable e
           (let [[instance args] (retry/retry instance args e)]
             (if (:retry instance)
               ((wrap-exception f) instance args)
               (deliver (:result instance) {:type :error
                                            :data e})))))))

(defn invoke-base
  "constructs a standard procedure
   
   (def -proc- {:handler +
                :result (promise)})
 
   (invoke-base -proc- [1 2 3])
 
   @(:result -proc-)
   => {:type :success, :data 6}"
  {:added "3.0"}
  [instance args]
  (let [result (apply (:handler instance) args)]
    (deliver (:result instance) {:type :success
                                 :data result})))

(defn procedure-invoke
  "the full invocation for the procedure, incorporating middleware and retry"
  {:added "3.0"}
  [{:keys [id-fn handler arglist time] :as procedure} & args]
  (let [_          (if (< (count arglist) (count args))
                     (throw (Exception.
                             (str "There should be less inputs than the arglist: " arglist))))
        ninputs    (max-inputs handler (count args))
        opts       (zipmap arglist args)
        opts       (if-let [t (:instance opts)]
                     (nested/merge-nested t (dissoc opts :instance))
                     opts)
        instance   (nested/merge-nested procedure opts)
        instance   (update-in instance [:timestamp]
                              (fn [t] (or t
                                          (time/now time))))
        instance   (update-in instance [:runtime]
                              (fn [rt] (or rt
                                           (atom {}))))
        thread     (volatile! {})
        nargs      (->> arglist
                        (take ninputs)
                        (map #(get instance %)))
        instance   (-> instance
                       (assoc :thread thread)
                       (assoc :input args)
                       (assoc :args  nargs)
                       (assoc :procedure procedure)
                       (map->ProcedureInstance))
        prn-registry (fn [f text]
                       (fn [inst args]
                         (prn text (:registry inst))
                         (f inst args)))]
    
    ((-> invoke-base
         wrap-exception
         middleware/wrap-instance
         middleware/wrap-cached
         middleware/wrap-timing
         middleware/wrap-registry
         middleware/wrap-callback
         middleware/wrap-mode
         middleware/wrap-timeout
         middleware/wrap-interrupt
         middleware/wrap-id)
     instance nargs)))

(defn procedure-display
  "displays the procedure"
  {:added "3.0"}
  [proc]
  (-> (select-keys proc [:name :mode :params :cached :runtime :arglist :retry :timeout])
      (->> (into {}))))

(defexecutive Procedure
  "creates a procedure for computation"
  {:added "3.0"}
  []
  {:type defrecord
   :tag "procedure"
   :invoke procedure-invoke
   :display procedure-display})

(defn procedure
  "creates a procedure for computation
 
   @((procedure {:name \"ID\"
                 :handler (fn [id params instance]
                            ; (println (-> instance :retry :count))
                            (if (= 5 (-> instance :retry :count))
                              (-> instance :retry :count)
                              (throw (Exception.))))
                 :retry {:handle [{:on #{Exception}
                                   :apply   (fn [state e])
                                  :limit   (fn [state count])
                                   :wait    (fn [state count])}]
                         :count 0
                         :state  {:a 1 :b 2}
                         :limit 10
                         :wait  100}}
                [:id :params :instance])
     \"ID\" {} {:mode :async :cached false})
   => 5"
  {:added "3.0"}
  ([tk arglist]
   (cond (fn? tk)
         (procedure {:handler tk} arglist)

         (check/hash-map? tk)
         (-> (assoc tk :arglist arglist)
             (nested/merge-nil-nested +default-settings+)
             (map->Procedure)))))

(definvoke invoke-intern-procedure
  "creates the form for defining the procedure
 
   (invoke-intern-procedure '-hello- {:mode :sync} '([] (Thread/sleep 1000)))"
  {:added "3.0"}
  [:method {:multi protocol.function/-invoke-intern
            :val  :procedure}]
  ([name config body]
   (invoke-intern-procedure :procedure name config body))
  ([_ name config body]
   (let [arglist  (:arglist config)
         arglists (fn/form-arglists body)]
     `(do (def ~name (procedure (merge {:handler (fn ~name ~@body)} ~config) ~arglist))
          (doto (var ~name)
            (alter-meta! merge ~config ~(if arglist
                                          `{:arglists (list ~arglist)}
                                          `{:arglists ~arglists})))))))

(defmacro defprocedure
  "defining a procedure
 
   (defprocedure -hello-
     {:mode :sync}
     ([]
      (Thread/sleep 1000)
      :DONE))
   
   (defprocedure -print-hello-
     {:id-fn :timestamp
      :arglist [:timestamp :params :instance]
     :params {:b 2}}
     ([t params instance]
      (println \"INSTANCE: \" instance)
      (Thread/sleep 500)
      (println \"ENDED\" t)))"
  {:added "3.0"}
  [name config & body]
  (invoke-intern-procedure :procedure name config body))

(defn procedure-kill
  "kills a running procedure
   (def -proc- ((procedure {:name \"hello\"
                            :id :1
                            :handler (fn [] (Thread/sleep 1000000000))} [])))
   
   (Thread/sleep 100)
   (procedure-kill -proc-)
   => true"
  {:added "3.0"}
  [{:keys [registry name id]}]
  (registry/kill registry name id))

(defn procedure-running?
  "checks if a procedure is running
 
   (def -proc- ((procedure {:name \"hello\"
                            :id :1
                            :handler (fn [] (Thread/sleep 1000000000))} [])))
 
   (procedure-running? -proc-)
   => true"
  {:added "3.0"}
  [{:keys [thread] :as p}]
  (and @thread
       (not (future-done? @thread))))
