(ns hara.code.framework
  (:require [clojure.walk :as walk]
            [hara.string :as string]
            [hara.string.base.ansi :as ansi]
            [hara.lib.diff :as diff]
            [hara.code.framework.cache :as cache]
            [hara.code.framework.common :as common]
            [hara.code.framework.docstring :as docstring]
            [hara.code.framework.test.clojure]
            [hara.code.framework.test.fact]
            [hara.code.framework.text :as text]
            [hara.code.query :as query]
            [hara.module :as module]
            [hara.core.base.result :as result]
            [hara.data.base.nested :as nested]
            [hara.io.file :as fs]
            [hara.io.project :as project]
            [hara.function.task :as task]
            [hara.code.navigate :as nav]))

(def ^:dynamic *toplevel-forms*
  '#{defn
     defmulti
     defmacro
     defcache
     definvoke
     defexecutive
     defmemoize
     deftask})

(defn import-selector
  "creates an import selector
 
   (import-selector '-hello-)
   ;;[(#{<options>} | -hello- string? map? & _)]
   => vector?"
  {:added "3.0"}
  ([] (import-selector '_))
  ([var]
   [(list *toplevel-forms* '| var '^:%?- string? '^:%?- map? '& '_)]))

(defn toplevel-selector
  "creates a selector for querying toplevel forms
 
   (toplevel-selector '-hello-)
   ;;[(#{<def forms>} | -hello- & _)]
   => vector?"
  {:added "3.0"}
  ([] (toplevel-selector '_))
  ([var]
   [(list *toplevel-forms* '| var '& '_)]))

(defn analyse-source-function
  "helper function for `analyse-source-code`"
  {:added "3.0"}
  [nsp nav]
  (let [id   (nav/value nav)
        line (-> nav nav/up nav/line-info)
        nav (if (= :string (nav/tag (nav/right nav)))
              (nav/delete-right nav)
              nav)
        nav (if (= :map (nav/tag (nav/right nav)))
              (nav/delete-right nav)
              nav)
        code (-> nav nav/up nav/string)]
    [id {:ns nsp
         :var id
         :source {:code code
                  :line line
                  :path common/*path*}}]))

(defn analyse-source-code
  "analyses a source file for namespace and function definitions
 
   (-> (analyse-source-code (slurp \"dev/test/hara.code/src/example/core.clj\"))
       (get-in '[example.core -foo-]))
   => '{:ns example.core,
       :var -foo-,
       :source {:code \"(defn -foo-\\n  [x]\\n  (println x \\\"Hello, World!\\\"))\",
                :line {:row 3, :col 1, :end-row 6, :end-col 31},
                :path nil}}"
  {:added "3.0"}
  [s]
  (let [nav  (-> (nav/parse-root s)
                 (nav/down))
        nsp  (->  (query/$ nav [(ns | _ & _)] {:walk :top})
                  first)
        fns  (->> (query/$* nav (toplevel-selector) {:return :zipper :walk :top})
                  (map (partial analyse-source-function nsp))
                  (into {}))]
    (common/entry {nsp fns})))

(defn find-test-frameworks
  "find test frameworks given a namespace form
   (find-test-frameworks '(ns ...
                       (:use hara.test)))
   => #{:fact}
 
   (find-test-frameworks '(ns ...
                       (:use clojure.test)))
   => #{:clojure}"
  {:added "3.0"}
  [ns-form]
  (let [folio (atom #{})]
    (walk/postwalk (fn [form]
                     (if-let [k (common/test-frameworks form)]
                       (swap! folio conj k)))
                   ns-form)
    @folio))

(defn analyse-test-code
  "analyses a test file for docstring forms
 
   (-> (analyse-test-code (slurp \"dev/test/hara.code/test/example/core_test.clj\"))
       (get-in '[example.core -foo-])
       (update-in [:test :code] docstring/->docstring))
   => (contains '{:ns example.core
                  :var -foo-
                  :test {:code \"1\\n  => 1\"
                         :line {:row 6 :col 1 :end-row 7 :end-col 16}
                         :path nil}
                 :meta {:added \"3.0\"}
                  :intro \"\"})"
  {:added "3.0"}
  [s]
  (let [nav        (-> (nav/parse-root s)
                       (nav/down))
        nsloc      (query/$ nav [(ns | _ & _)] {:walk :top
                                                :return :zipper
                                                :first true})
        
        nsp        (nav/value nsloc)
        ns-form    (-> nsloc nav/up nav/value)
        frameworks (find-test-frameworks ns-form)]
    (->> frameworks
         (map (fn [framework] (common/analyse-test framework nav)))
         (apply nested/merge-nested)
         (common/entry))))

(def ^{:arglists '([type path time])}
  analyse-file-fn
  (memoize (fn [type path time]
             (case type
               :source (analyse-source-code (slurp path))
               :test   (analyse-test-code (slurp path))))))

(defn analyse-file
  "helper function for analyse, taking a file as input"
  {:added "3.0"}
  ([path]
   (analyse-file (project/file-type path) path))
  ([type path]
   (binding [common/*path* path]
     (analyse-file-fn type path (-> (fs/path path)
                                    (fs/attributes)
                                    (:last-modified-time))))))

(defn analyse
  "seed analyse function for the `hara.code/analyse` task
   
   (->> (project/in-context (analyse 'hara.code.framework-test))
        (common/display-entry))
   => (contains-in {:test {'hara.code.framework
                           (contains '[analyse
                                       analyse-file
                                       analyse-source-code])}})
   
   (->> (project/in-context (analyse 'hara.code.framework))
        (common/display-entry))
   => (contains-in {:source {'hara.code.framework
                             (contains '[analyse
                                         analyse-file
                                         analyse-source-code])}})"
  {:added "3.0"}
  [ns {:keys [output sorted] :as params} lookup project]
  (let [path    (lookup ns)
        result  (cache/fetch ns path)
        nresult (or result
                    (and path
                         (analyse-file path)))
        _    (if (nil? result)
               (cache/update ns nresult))
        nresult (cond (and (nil? nresult)
                           (nil? path))
                      (cond (= type :test)
                            (result/result {:status :error
                                            :data :no-test-file})
                            
                            (= type :source)
                            (result/result {:status :error
                                            :data :no-test-file}))
                      
                      (= output :map)
                      nresult
                      
                      (= output :vector)
                      (let [nresult (mapcat vals (vals nresult))]
                        (if (true? sorted)
                          (sort-by :var nresult)
                          (sort-by #(or (-> % :source :line :row)
                                        (-> % :test :line :row)) nresult)))

                      :else
                      nresult)]
    nresult))

(defn var-function
  "constructs a var, with or without namespace
 
   ((var-function true) {:ns 'hello :var 'world})
   => 'hello/world
 
   ((var-function false) {:ns 'hello :var 'world})
   => 'world"
  {:added "3.0"}
  [full]
  (fn [{:keys [ns var] :as m}]
    (with-meta (if (true? full)
                 (symbol (str ns) (str var))
                 var)
      m)))

(defn vars
  "returns all vars in a given namespace
   (project/in-context (vars {:sorted true}))
   => (contains '[analyse
                  analyse-file
                  analyse-source-code
                  analyse-source-function
                  ])"
  {:added "3.0"}
  [ns {:keys [full sorted] :as params} lookup project]
  (let [analysis (analyse ns {:output :vector :sorted sorted} lookup project)]
    (cond (result/result? analysis)
          analysis
          
          :else
          (mapv (var-function full) analysis))))

(defn docstrings
  "returns all docstrings in a given namespace with given keys
 
   (->> (project/in-context (docstrings))
        (map first)
        sort)
   => (project/in-context (vars {:sorted true}))"
  {:added "3.0"}
  [ns {:keys [full sorted]} lookup project]
  (let [ns       (project/test-ns ns)
        analysis (analyse ns {:output :vector :sorted sorted} lookup project)]
    (cond (result/result? analysis) analysis

          :else
          (->> analysis
               (map (juxt (var-function full)
                          (comp docstring/->docstring :code :test)))
               (into {})))))

(defn transform-code
  "transforms the code and performs a diff to see what has changed
 
   ;; options include :skip, :full and :write
   (project/in-context (transform-code 'hara.code.framework {:transform identity}))
   => (contains {:changed []
                 :updated false
                 :path any})"
  {:added "3.0"}
  [ns {:keys [write print skip transform full] :as params} lookup {:keys [root] :as project}]
  (cond skip
        (let [path  (lookup ns)
              text  (if (fs/exists? path) "" (slurp path))]
          (spit path (transform text))
          {:updated true :path path})

        :else
        (let [path     (lookup ns)
              params  (assoc params :output :map)
              [original analysis] (if (fs/exists? path)
                                    [(slurp path) (analyse ns params lookup project)]
                                    ["" {}])
              revised  (transform original)
              deltas   (text/deltas ns analysis original revised)
              path     (str (fs/relativize root path))
              updated  (when (and write (seq deltas))
                         (spit path revised)
                         true)
              _        (when (and (:function print) (seq deltas))
                         (clojure.core/print (str "\n" (ansi/style path #{:bold :blue :underline}) "\n\n"))
                         (clojure.core/print (diff/->string deltas) "\n"))]
          (cond-> (text/summarise-deltas deltas)
            full  (assoc :deltas deltas)
            :then (assoc :updated (boolean updated)
                         :path (str (fs/relativize root path)))))))

(defn locate-code
  "finds code base upon a query
 
   (project/in-context (locate-code {:query '[docstrings]
                                     :print {:function true}}))"
  {:added "3.0"}
  [ns {:keys [print query process] :as params} lookup {:keys [root] :as project}]
  (let [path     (lookup ns)
        code     (slurp path)
        analysis (analyse ns params lookup project)
        line-lu  (common/line-lookup ns analysis)
        locs     (atom [])
        results  (->> (query/$* (nav/parse-root code)
                                query
                                {:return :zipper})
                      (mapv nav/line-info))]
    (if (seq results)
      (let [lines (string/split-lines code)
            path  (str (fs/relativize root path))]
        (when (:function print)
          (clojure.core/print (str "\n" (ansi/style path #{:bold :blue :underline}) "\n\n"))
          (clojure.core/print (text/->string results lines line-lu params) "\n"))
        results))))

(defn- all-string [nav]
  (->> (iterate nav/right* nav)
       (take-while identity)
       (map nav/string)
       (string/joinl)))

(defn refactor-code
  "takes in a series of edits and performs them on the code
 
   (project/in-context (refactor-code {:edits []}))
   => {:changed [], :updated false, :path \"test/hara/code/framework_test.clj\"}"
  {:added "3.0"}
  [ns {:keys [edits] :as params} lookup project]
  (let [;;funcs (map +transform+ edits)
        transform (fn [original]
                    (reduce (fn [text transform]
                              (all-string (transform (nav/parse-root text))))
                            original
                            edits))
        params  (assoc params :transform transform)]
    (transform-code ns params lookup project)))

(comment
  (project/in-context (locate-code 'hara.code.format.ns
                                   {:query hara.code.format.ns/+load-has-shorthand+
                                    :print {:function true}}))
  
  (project/in-context (locate-parent 'hara.code.format.ns
                                     {:query hara.code.format.ns/+load-has-shorthand+
                                      :print {:function true}}))
  
  (project/in-context (locate-parent 'hara.code.format.ns
                                     {:query hara.code.format.ns/+ns-has-use-form+
                                      :print {:function true}}))

  (project/in-context (locate-parent 'hara.code.format.ns
                                     {:query hara.code.format.ns/+require-has-refer+
                                      :print {:function

                                              true}})))

(comment
  
  (./reset '[hara.block hara.code])
  (def a* (analyse-source-code (slurp "src/hara/code/framework.clj")))
  
  (def a* (analyse-test-code (slurp "test/hara/code/framework_test.clj")))
  
  
  (-> (nav/parse-root (slurp "test/hara/code/base_test.clj"))
      (query/select [{:is 'docstrings}]
                    )
      (first)
      )
  
  (-> (nav/parse-root (slurp "test/hara/code/base_test.clj"))
      (query/select [{:is 'docstrings}]
                    )
      (first)
      (nav/up)
      (nav/up)
      (nav/up)
      (nav/up)
      (nav/up)
      (nav/left)
      (nav/left)
      (nav/left)
      (nav/left)
      (nav/left)
      (nav/find-next-token :code)
      (nav/find-next-token :code)
      (nav/next)
      ;(nav/right-expression)
      (nav/height))
  
  13
  
  (-> (nav/parse-root (slurp "dev/test/hara.code/src/example/core.clj"))
      (query/select [{:is 'foo}]
                    )
      )
  )
