(ns hara.code.block.type
  (:require [hara.protocol.block :as protocol.block]
            [hara.code.block.base :as base]
            [hara.code.block.check :as check]
            [hara.code.block.value :as value]
            [hara.string :as string]))

(def ^:dynamic *tab-width* 4)

(defn block-compare
  "compares equality of two blocks
 
   (block-compare (construct/void \\space)
                  (construct/void \\space))
   => 0"
  {:added "3.0"}
  [this other]
  (+ (.compareTo (base/block-tag this)
                 (base/block-tag other))
     (.compareTo (base/block-string this)
                 (base/block-string other))))

(deftype VoidBlock [tag ^Character character width height]

  protocol.block/IBlock
  (-type     [_] :void)
  (-tag      [_] tag)
  (-string   [_] (str character))
  (-length   [_] 1)
  (-width    [_] width)
  (-height   [_] height)
  (-prefixed [_] 0)
  (-suffixed [_] 0)
  (-verify   [_] (and tag (= tag (check/void-tag character)))) 

  Comparable
  (compareTo [this other]
    (block-compare this other))
  
  Object
  (toString [_]
    (str (or (base/*void-representations* character) character))))

(defmethod print-method VoidBlock
  [v w]
  (.write w (str v)))

(defn void-block?
  "checks if the block is a void block
 
   (void-block? (construct/void))
   => true"
  {:added "3.0"}
  [block]
  (instance? VoidBlock block))

(defn void-block
  "constructs a void block
 
   (-> (void-block :linespace \\tab 1 0)
       (base/block-info))
   => {:type :void, :tag :linespace, :string \"\\t\", :height 0, :width 1}"
  {:added "3.0"}
  [tag ch width height]
  (->VoidBlock tag ch width height))

(defn space-block?
  "checks if block is of type \\space
 
   (space-block? (construct/space))
   => true"
  {:added "3.0"}
  [block]
  (and (void-block? block)
       (= (.-character block) \space)))

(defn linebreak-block?
  "checks if block is of type :linebreak
 
   (linebreak-block? (construct/newline))
   => true"
  {:added "3.0"}
  [block]
  (and (void-block? block)
       (= :linebreak (base/block-tag block))))

(defn linespace-block?
  "checks if block is of type :linespace
 
   (linespace-block? (construct/space))
   => true"
  {:added "3.0"}
  [block]
  (and (void-block? block)
       (= :linespace (base/block-tag block))))

(defn eof-block?
  "checks if input is an eof block
 
   (eof-block? (construct/void nil))
   => true"
  {:added "3.0"}
  [block]
  (and (void-block? block)
       (= :eof (base/block-tag block))))

(defn nil-void?
  "checks if block is nil or type void block
 
   (nil-void? nil) => true
 
   (nil-void? (construct/block nil)) => false
 
   (nil-void? (construct/space)) => true"
  {:added "3.0"}
  [block]
  (or (nil? block)
      (void-block? block)))

(deftype CommentBlock [string width]
  
  protocol.block/IBlock
  (-type   [_] :comment)
  (-tag    [_] :comment)
  (-string [_] string)
  (-length [_] width)
  (-width  [_] width)
  (-height [_] 0)
  (-prefixed [_] 0)
  (-suffixed [_] 0)
  (-verify [_] (check/comment? string))

  Comparable
  (compareTo [this other]
    (block-compare this other))
  
  Object
  (toString [_] string))

(defmethod print-method CommentBlock
  [v w]
  (.write w (str v)))

(defn comment-block?
  "checks if the block is a token block
 
   (comment-block? (construct/comment \";;hello\"))
   => true"
  {:added "3.0"}
  [block]
  (instance? CommentBlock block))

(defn comment-block
  "constructs a comment block
 
   (-> (comment-block \";hello\")
       (base/block-info))
   => {:type :comment, :tag :comment, :string \";hello\", :height 0, :width 6}"
  {:added "3.0"}
  [string]
  (->CommentBlock string (count string)))

(deftype TokenBlock [tag string value value-string width height]
  
  protocol.block/IBlock
  (-type   [_] :token)
  (-tag    [_] tag)
  (-string [_] string)
  (-length [_] (count string))
  (-width  [_] width)
  (-height [_] height)
  (-prefixed [_] 0)
  (-suffixed [_] 0)
  (-verify [_] (and tag (= tag (check/token-tag value))))

  protocol.block/IBlockExpression
  (-value  [_] value)
  (-value-string [_] value-string)
  
  Comparable
  (compareTo [this other]
    (block-compare this other))
  
  Object
  (toString [_] (pr-str value)))

(defmethod print-method TokenBlock
  [v w]
  (.write w (str v)))

(defn token-block?
  "checks if the block is a token block
 
   (token-block? (construct/token \"hello\"))
   => true"
  {:added "3.0"}
  [block]
  (instance? TokenBlock block))

(defn token-block
  "creates a token block
   
   (base/block-info (token-block :symbol \"abc\" 'abc \"abc\" 3 0))
   => {:type :token, :tag :symbol, :string \"abc\", :height 0, :width 3}"
  {:added "3.0"}
  [tag string value value-string width height]
  (->TokenBlock tag string value value-string width height))

(defn container-width
  "calculates the width of a container
 
   (container-width (construct/block [1 2 3 4]))
   => 9"
  {:added "3.0"}
  [block]
  (loop [total (base/block-suffixed block)
         [child & more] (reverse (base/block-children block))]
    (cond (nil? child)
          (+ total
             (base/block-prefixed block))

          (linebreak-block? child)
          total

          (zero? (base/block-height child))
          (recur (+ total (base/block-width child))
                 more)
          
          (base/container? child)
          (+ total (base/block-width child))

          :else total)))

(defn container-height
  "calculates the height of a container
 
   (container-height (construct/block [(construct/newline)
                                       (construct/newline)]))
   => 2"
  {:added "3.0"}
  [block]
  (reduce (fn [total child]
            (+ total (base/block-height child)))
          0
          (base/block-children block)))

(defmulti container-string
  "returns the string for the container
 
   (container-string (construct/block [1 2 3]))
   => \"[1 2 3]\""
  {:added "3.0"}
  base/block-tag)

(defmethod container-string :root
  [block]
  (->> (base/block-children block)
       (map base/block-string)
       (apply str)))

(defmethod container-string :default
  [block]
  (let [{:keys [start end]} (.-props block)
        children (.-children block)
        all (apply str (map base/block-string children))]
    (str start all end)))

(defn container-value-string
  "returns the string for 
 
   (container-value-string (construct/block [::a :b :c]))
   => \"[:hara.code.block.type-test/a :b :c]\"
 
   (container-value-string (parse/parse-string \"[::a :b :c]\"))
   => \"[(keyword \\\":a\\\") (keyword \\\"b\\\") (keyword \\\"c\\\")]\""
  {:added "3.0"}
  [block]
  (let [{:keys [start end]} (.-props block)
        children (.children block)
        all  (->> (keep (fn [block]
                          (or (if (base/expression? block)
                                (base/block-value-string block))
                              (if (base/modifier? block)
                                (base/block-string block))))
                        children)
                  (string/join " "))]
    (str start all end)))

(defrecord ContainerBlock [tag children props]
  
  protocol.block/IBlock
  (-type     [_] :collection)
  (-tag      [_] tag)
  (-string   [this] (container-string this))
  (-length   [this] (count (base/block-string this)))
  (-width    [this] (container-width this))
  (-height   [this] (container-height this))
  (-prefixed [_] (count (:start props)))
  (-suffixed [_] (count (:end props)))
  
  Comparable
  (compareTo [this other]
    (block-compare this other))

  Object
  (toString [obj] (base/block-string obj))
  
  protocol.block/IBlockExpression
  (-value [block] ((:value props) block))
  (-value-string [block] (container-value-string block))

  protocol.block/IBlockContainer
  (-children [_] children)
  (-replace-children [_ children]
    (ContainerBlock. tag children props)))

(defmethod print-method ContainerBlock
  [v w]
  (.write w (str v)))

(defn container-block?
  "checks if block is a container block
 
   (container-block? (construct/block []))
   => true"
  {:added "3.0"}
  [block]
  (instance? ContainerBlock block))

(defn container-block
  "constructs a container block
 
   (-> (container-block :fn [(construct/token '+)
                            (construct/void)
                            (construct/token '1)]
                       (construct/*container-props* :fn))
       (base/block-value))
   => '(fn* [] (+ 1))"
  {:added "3.0"}
  [tag children props]
  (->ContainerBlock tag children props))

(defrecord ModifierBlock [tag string command]
  
  protocol.block/IBlock
  (-type     [_] :modifier)
  (-tag      [_] tag)
  (-string   [_] string)
  (-length   [_] (count string))
  (-width    [_] (count string))
  (-height   [_] 0)
  (-prefixed [_] 0)
  (-suffixed [_] 0)

  Comparable
  (compareTo [this other]
    (block-compare this other))
  
  Object
  (toString [obj] string)

  protocol.block/IBlockModifier
  (-modify [_ accumulator input]
    (command accumulator input)))

(defmethod print-method ModifierBlock
  [v w]
  (.write w (str v)))

(defn modifier-block?
  "checks if block is a modifier block
 
   (modifier-block? (construct/uneval))
   => true"
  {:added "3.0"}
  [block]
  (instance? ModifierBlock block))

(defn modifier-block
  "creates a modifier block, specifically #_
 
   (modifier-block :hash-uneval \"#_\" (fn [acc _] acc))"
  {:added "3.0"}
  [tag string command]
  (->ModifierBlock tag string command))
