(ns hara.code.navigate
  (:require [hara.code.block.base :as base]
            [hara.code.block.construct :as construct]
            [hara.code.block.type :as type]
            [hara.code.block.parse :as parse]
            [hara.core.zip :as zip]
            [hara.module :as module])
  (:refer-clojure :exclude [next replace type]))

(defn left-anchor
  "calculates the length to the last newline
   
   (left-anchor (-> (navigator nil)
                    (zip/step-right)))
   => 3"
  {:added "3.0"}
  [nav]
  (cond (and (zip/at-outside-most? nav)
             (zip/at-left-most? nav))
        0
        
        :else
        (loop [total 0
               [lblock & more] (:left nav)]
          (cond (and (nil? lblock)
                     (zip/at-outside-most? nav))
                total

                (nil? lblock)
                (let [pnav (zip/step-outside nav)]
                  (+ total
                     (if-let [elem (zip/right-element pnav)] 
                       (base/block-prefixed elem)
                       0)
                     (left-anchor pnav)))

                (zero? (base/block-height lblock))
                (recur (+ total (base/block-width lblock))
                       more)

                (base/container? lblock)
                (+ total (base/block-width lblock))
                
                :else total))))

(defn update-step-left
  "updates the step position to the left
 
   (-> {:position [0 7]}
       (update-step-left (construct/block [1 2 3])))
   => {:position [0 0]}"
  {:added "3.0"}
  [nav lblock]
  (update-in nav
             [:position]
             (fn [[row col]]
               (let [rows    (base/block-height lblock)
                     cols    (base/block-width lblock)]
                 (if (pos? rows)
                   [(- row rows) (left-anchor nav)]
                   [row (- col cols)])))))

(defn update-step-right
  "updates the step position to the right
 
   (-> {:position [0 0]}
       (update-step-right (construct/block [1 2 3])))
   => {:position [0 7]}"
  {:added "3.0"}
  [nav block]
  (-> (update-in nav
                 [:position]
                 (fn [[row col]]
                   (let [rows (base/block-height block)
                         cols (base/block-width block)]
                     (if (pos? rows)
                       [(+ row rows) cols]
                       [row (+ col cols)]))))))

(defn update-step-inside
  "updates the step position to within a block
   (-> {:position [0 0]}
       (update-step-inside (construct/block #{})))
   => {:position [0 2]}"
  {:added "3.0"}
  [nav block]
  (update-in nav
             [:position]
             (fn [[row col]]
               [row (+ col (base/block-prefixed block))])))

(defn update-step-inside-left
  "updates the step position to within a block
   (-> {:position [0 3]}
       (update-step-inside-left (construct/block #{})))
   => {:position [0 2]}"
  {:added "3.0"}
  [nav block]
  (update-in nav
             [:position]
             (fn [[row col]]
               [row (- col (base/block-suffixed block))])))

(defn update-step-outside
  "updates the step position to be outside a block
 
   (let [left-elems [(construct/block [1 2 3]) (construct/newline)]]
     (-> {:position [1 0]
          :left left-elems}
         (update-step-outside left-elems)
         :position))
   => [0 7]"
  {:added "3.0"}
  [nav left-blocks]
  (update-in nav
             [:position]
             (fn [[row col]]
               (let [lrows   (reduce (fn [total block]
                                       (+ total (base/block-height block)))
                                     0
                                     left-blocks)]
                 (cond (zero? lrows)
                       [row (- col
                               (base/block-prefixed (zip/right-element nav))
                               (apply + (map base/block-width left-blocks)))]
                       
                       :else
                       [(- row lrows)
                        (left-anchor nav)])))))

(defn display-navigator
  "displays a string representing the navigator
 
   (-> (navigator [1 2 3 4])
       (display-navigator))
   => \"<0,0> |[1 2 3 4]\""
  {:added "3.0"}
  [nav]
  (let [[row col] (:position nav)]
    (format "<%d,%d> %s"
            row
            col
            (apply str (zip/status nav)))))

(defn navigator
  "creates a navigator for the block
 
   (str (navigator [1 2 3 4]))
   => \"<0,0> |[1 2 3 4]\""
  {:added "3.0"}
  ([block]
   (navigator block {}))
  ([block opts]
   (zip/zipper (construct/block block)
               (merge zip/+base+
                      {:create-container        construct/empty
                       :create-element          construct/block
                       :cursor                  (construct/cursor)   
                       :is-container?           base/container?
                       :is-empty-container?     (comp empty? base/block-children)
                       :is-element?             base/block?
                       :list-elements           base/block-children
                       :add-element             construct/add-child
                       :update-elements         base/replace-children
                       :update-step-left        update-step-left
                       :update-step-right       update-step-right
                       :update-step-inside      update-step-inside
                       :update-step-inside-left update-step-inside-left
                       :update-step-outside     update-step-outside
                       :update-delete-left      update-step-left
                       :update-delete-right     (fn [zip elem] zip)
                       :update-insert-left      update-step-right
                       :update-insert-right     (fn [zip elem] zip)})
               {:tag :navigator
                :prefix ""
                :display display-navigator
                :position [0 0]})))

(defn navigator?
  "checks if object is navigator
 
   (navigator? (navigator [1 2 3 4]))
   => true"
  {:added "3.0"}
  [obj]
  (and (instance? hara.core.zip.Zipper obj)
       (= (:tag obj) :navigator)))

(defn from-status
  "constructs a navigator from a given status
 
   (str (from-status (construct/block [1 2 3 (construct/cursor) 4])))
   => \"<0,7> [1 2 3 |4]\""
  {:added "3.0"}
  [block]
  (zip/from-status block
                   navigator))

(defn parse-string
  "creates a navigator from string
 
   (str (parse-string \"(2   #|   3  )\"))
   => \"<0,5> (2   |   3  )\""
  {:added "3.0"}
  [string]
  (from-status (parse/parse-string string)))

(defn parse-root
  "parses the navigator from root string
 
   (str (parse-root \"a b c\"))
   => \"<0,0> |a b c\""
  {:added "3.0"}
  [string]
  (navigator (parse/parse-root string)))

(defn parse-root-status
  "parses string and creates a navigator from status
 
   (str (parse-root-status \"a b #|c\"))
   => \"<0,6> a b |c\""
  {:added "3.0"}
  [string]
  (from-status (parse/parse-root string)))

(defn root-string
  "returns the top level string
 
   (root-string (navigator [1 2 3 4]))
   => \"[1 2 3 4]\""
  {:added "3.0"}
  [nav]
  (->> nav
       (zip/step-outside-most)
       (zip/step-left-most)
       (zip/right-elements)
       (map base/block-string)
       (apply str)))

(defn left-expression
  "returns the expression on the left
   
   (-> {:left [(construct/newline)
               (construct/block [1 2 3])]}
       (left-expression)
       (base/block-value))
   => [1 2 3]"
  {:added "3.0"}
  [nav]
  (->> (:left nav)
       (filter base/expression?)
       first))

(defn left-expressions
  "returns all expressions on the left
 
   (->> {:left [(construct/newline)
                (construct/block :b)
                (construct/space)
                (construct/space)
                (construct/block :a)]}
        (left-expressions)
        (mapv base/block-value))
   => [:a :b]"
  {:added "3.0"}
  [nav]
  (->> (zip/left-elements nav)
       (filter base/expression?)))

(defn right-expression
  "returns the expression on the right
 
   (-> {:right [(construct/newline)
               (construct/block [1 2 3])]}
       (right-expression)
       (base/block-value))
   => [1 2 3]"
  {:added "3.0"}
  [nav]
  (->> (zip/right-elements nav)
       (filter base/expression?)
       first))

(defn right-expressions
  "returns all expressions on the right
 
   (->> {:right [(construct/newline)
                (construct/block :b)
                (construct/space)
                (construct/space)
                (construct/block :a)]}
        (right-expressions)
        (mapv base/block-value))
   => [:b :a]"
  {:added "3.0"}
  [nav]
  (->> (zip/right-elements nav)
       (filter base/expression?)))

(defn left
  "moves to the left expression
 
   (-> (parse-string \"(1  [1 2 3]    #|)\")
       (left)
       str)
   => \"<0,4> (1  |[1 2 3]    )\""
  {:added "3.0"}
  [nav]
  (zip/find-left nav base/expression?))

(defn left-most
  "moves to the left-most expression
 
   (-> (parse-string \"(1  [1 2 3]  3 4   #|)\")
       (left-most)
       str)
   => \"<0,1> (|1  [1 2 3]  3 4   )\""
  {:added "3.0"}
  [nav]
  (if (nil? (left nav))
    nav
    (recur (left nav))))

(defn left-most?
  "checks if navigator is at left-most
   
   (-> (from-status [1 [(construct/cursor) 2 3]])
       (left-most?))
   => true"
  {:added "3.0"}
  [nav]
  (nil? (left nav)))

(defn right
  "moves to the expression on the right
   
   (-> (parse-string \"(#|[1 2 3]  3 4  ) \")
       (right)
       str)
   => \"<0,10> ([1 2 3]  |3 4  )\""
  {:added "3.0"}
  [nav]
  (zip/find-right nav base/expression?))

(defn right-most
  "moves to the right-most expression
 
   (-> (parse-string \"(#|[1 2 3]  3 4  ) \")
       (right-most)
       str)
   => \"<0,12> ([1 2 3]  3 |4  )\""
  {:added "3.0"}
  [nav]
  (if (nil? (right nav))
    nav
    (recur (right nav))))

(defn right-most?
  "checks if navigator is at right-most
   
   (-> (from-status [1 [2 3 (construct/cursor)]])
       (right-most?))
   => true"
  {:added "3.0"}
  [nav]
  (nil? (right nav)))

(defn up
  "navigates outside of the form
 
   (str (up (from-status [1 [2 (construct/cursor) 3]])))
   => \"<0,3> [1 |[2 3]]\""
  {:added "3.0"}
  [nav]
  (zip/step-outside nav))

(defn down
  "navigates into the form
 
   (str (down (from-status [1 (construct/cursor) [2 3]])))
   => \"<0,4> [1 [|2 3]]\""
  {:added "3.0"}
  [nav]
  (if (zip/can-step-inside? nav)
    (zip/step-inside nav)))

(defn right*
  "navigates to right element, including whitespace
 
   (str (right* (from-status [(construct/cursor) 1 2])))
   => \"<0,2> [1| 2]\""
  {:added "3.0"}
  [nav]
  (zip/step-right nav))

(defn left*
  "navigates to left element, including whitespace
 
   (str (left* (from-status [1 (construct/cursor) 2])))
   => \"<0,2> [1| 2]\""
  {:added "3.0"}
  [nav]
  (zip/step-left nav))

(defn block
  "returns the current block
 
   (block (from-status [1 [2 (construct/cursor) 3]]))
   => (construct/block 3)"
  {:added "3.0"}
  [nav]
  (zip/get nav))

(defn prev
  "moves to the previous expression
 
   (-> (parse-string \"([1 2 [3]] #|)\")
       (prev)
       str)
   => \"<0,7> ([1 2 [|3]] )\""
  {:added "3.0"}
  [nav]
  (zip/find-prev nav base/expression?))

(defn next
  "moves to the next expression
 
   (-> (parse-string \"(#|  [[3]]  )\")
       (next)
       (next)
       (next)
       str)
   => \"<0,5> (  [[|3]]  )\""
  {:added "3.0"}
  [nav]
  (zip/find-next nav base/expression?))

(defn find-next-token
  "moves tot the next token
 
   (-> (parse-string \"(#|  [[3 2]]  )\")
       (find-next-token 2)
       str)
   => \"<0,7> (  [[3 |2]]  )\""
  {:added "3.0"}
  [nav data]
  (zip/find-next nav (fn [block]
                       (= data (base/block-value block)))))

(defn prev-anchor
  "moves to the previous newline
 
   (-> (parse-string \"( \\n \\n [[3 \\n]] #|  )\")
       (prev-anchor)
       (:position))
   => [3 0]
 
   (-> (parse-string \"( #| )\")
       (prev-anchor)
       (:position))
   => [0 0]"
  {:added "3.0"}
  [nav]
  (if-let [nav (and nav
                    (zip/find-prev (zip/step-left nav)
                                   type/linebreak-block?))]
    (zip/step-right nav)
    (-> nav
        (zip/step-outside-most)
        (zip/step-left-most))))

(defn next-anchor
  "moves to the next newline
 
   (-> (parse-string \"( \\n \\n#| [[3 \\n]]  )\")
       (next-anchor)
       (:position))
   => [3 0]"
  {:added "3.0"}
  [nav]
  (if-let [nav (and nav (zip/find-next nav type/linebreak-block?))]    
    (zip/step-right nav)))

(defn left-token
  "moves to the left token
 
   (-> (parse-string \"(1  {}  #|2 3 4)\")
       (left-token)
       str)
   => \"<0,1> (|1  {}  2 3 4)\""
  {:added "3.0"}
  [nav]
  (zip/find-left nav type/token-block?))

(defn left-most-token
  "moves to the left-most token
 
   (-> (parse-string \"(1  {}  2 3 #|4)\")
       (left-most-token)
       str)
   \"<0,10> (1  {}  2 |3 4)\"
   "
  {:added "3.0"}
  [nav]
  (if (nil? (left-token nav))
    nav
    (recur (left-token nav))))

(defn right-token
  "moves to the right token
 
   (-> (parse-string \"(#|1  {}  2 3 4)\")
       (right-token)
       str)
   => \"<0,8> (1  {}  |2 3 4)\""
  {:added "3.0"}
  [nav]
  (zip/find-right nav type/token-block?))

(defn right-most-token
  "moves to the right-most token
 
   (-> (parse-string \"(#|1  {}  2 3 [4])\")
       (right-most-token)
       str)
   => \"<0,10> (1  {}  2 |3 [4])\""
  {:added "3.0"}
  [nav]
  (if (nil? (right-token nav))
    nav
    (recur (right-token nav))))

(defn prev-token
  "moves to the previous token
 
   (-> (parse-string \"(1 (2 3 [4])#|)\")
       (prev-token)
       str)
   => \"<0,9> (1 (2 3 [|4]))\""
  {:added "3.0"}
  [nav]
  (zip/find-prev nav type/token-block?))

(defn next-token
  "moves to the next token
 
   (-> (parse-string \"(#|[[1 2 3 4]])\")
       (next-token)
       str)
   => \"<0,3> ([[|1 2 3 4]])\""
  {:added "3.0"}
  [nav]
  (zip/find-next nav type/token-block?))

(defn position-left
  "moves the cursor to left expression
   
   (-> (parse-string \"( 2   #|   3  )\")
       (position-left)
       str)
   => \"<0,2> ( |2      3  )\""
  {:added "3.0"}
  [nav]
  (cond (base/expression? (zip/right-element nav))
        nav

        (empty? (left-expressions nav))
        (zip/step-left-most nav)

        :else
        (left nav)))

(defn position-right
  "moves the cursor the right expression
 
   (-> (parse-string \"(2   #|    3  )\")
       (position-right)
       str)
   => \"<0,9> (2       |3  )\""
  {:added "3.0"}
  [nav]
  (cond (base/expression? (zip/right-element nav))
        nav

        (empty? (right-expressions nav))
        (zip/step-right-most nav)

        :else
        (right nav)))

(defn tighten-left
  "removes extra spaces on the left
 
   (-> (parse-string \"(1 2 3   #|4)\")
       (tighten-left)
       str)
   => \"<0,7> (1 2 3 |4)\""
  {:added "3.0"}
  [nav]
  (loop [nav  (position-right nav)]
    (let [elem (zip/left-element nav)]
      (cond (nil? elem)
            nav

            (type/void-block? elem)
            (recur (zip/delete-left nav))

            (type/comment-block? elem)
            (zip/insert-left nav (construct/newline))

            :else
            (zip/insert-left nav (construct/space))))))

(defn tighten-right
  "removes extra spaces on the right
 
   (-> (parse-string \"(1 2 #|3       4)\")
       (tighten-right)
       str)
   => \"<0,5> (1 2 |3 4)\""
  {:added "3.0"}
  [nav]
  (loop [nav (position-left nav)]
    (let [elem (zip/right-element (zip/step-right nav))]
      (cond (nil? elem)
            (if (type/void-block? (zip/right-element nav))
              (zip/delete-right nav)
              nav)
            
            (type/void-block? elem)
            (-> (zip/step-right nav)
                (zip/delete-right)
                (zip/step-left)
                (recur))
            
            :else
            (-> (zip/step-right nav)
                (zip/insert-right (construct/space))
                (zip/step-left))))))

(defn tighten
  "removes extra spaces on both the left and right
 
   (-> (parse-string \"(1 2      #|3       4)\")
       (tighten)
       str)
   => \"<0,5> (1 2 |3 4)\""
  {:added "3.0"}
  [nav]
  (-> nav
      (tighten-left)
      (tighten-right)))

(defn level-empty?
  "checks if current container has expressions
 
   (-> (parse-string \"( #| )\")
       (level-empty?))
   => true"
  {:added "3.0"}
  [nav]
  (empty? (filter base/expression? (zip/current-elements nav))))

(defn insert-empty
  "inserts an element into an empty container
 
   (-> (parse-string \"( #| )\")
       (insert-empty 1)
       str)
   => \"<0,1> (|1  )\""
  {:added "3.0"}
  [nav data]
  (-> nav
      (zip/step-left-most)
      (zip/insert-right data)))

(defn insert-right
  "inserts an element to the right
   
   (-> (parse-string \"(#|0)\")
       (insert-right 1)
       str)
   => \"<0,1> (|0 1)\""
  {:added "3.0"}
  [nav data]
  (cond (level-empty? nav)
        (insert-empty nav data)

        :else
        (let [nav (position-right nav)]
          (-> nav
              (zip/step-right)
              (zip/insert-right data)
              (zip/insert-right (construct/space))
              (zip/step-left)))))

(defn insert-left
  "inserts an element to the left
   
   (-> (parse-string \"(#|0)\")
       (insert-left 1)
       str)
   => \"<0,3> (1 |0)\""
  {:added "3.0"}
  [nav data]
  (cond (level-empty? nav)
        (insert-empty nav data)

        :else
        (let [nav (position-left nav)]
          (-> nav
              (zip/insert-left data)
              (zip/insert-left (construct/space))))))

(defn insert
  "inserts an element and moves 
   
   (-> (parse-string \"(#|0)\")
       (insert 1)
       str)
   => \"<0,3> (0 |1)\""
  {:added "3.0"}
  [nav data]
  (cond (level-empty? nav)
        (insert-empty nav data)
        
        :else
        (let [nav (position-left nav)]
          (-> nav
              (zip/step-right)
              (zip/insert-left (construct/space))
              (zip/insert-right data)))))

(defn delete-left
  "deletes left of the current expression
   
   (-> (parse-string \"(1 2   #|3)\")
       (delete-left)
       str)
   => \"<0,3> (1 |3)\""
  {:added "3.0"}
  [nav]
  (cond (level-empty? nav)
        (tighten-left nav)
        
        :else
        (loop [nav (position-right nav)]
          (let [elem (zip/left-element nav)]
            (cond (nil? elem)
                  nav
                  
                  (base/expression? elem)
                  (zip/delete-left nav)
                  
                  :else
                  (recur (zip/delete-left nav)))))))

(defn delete-right
  "deletes right of the current expression
 
   (-> (parse-string \"(  #|1 2 3)\")
       (delete-right)
       str)
   => \"<0,3> (  |1 3)\""
  {:added "3.0"}
  [nav]
  (cond (level-empty? nav)
        (tighten-right nav)
        
        :else
        (loop [nav (position-left nav)]
          (let [elem (zip/right-element (zip/step-right nav))]
            (cond (nil? elem)
                  (if (type/void-block? (zip/right-element nav))
                    (zip/delete-right nav)
                    nav)
                  
                  (type/void-block? elem)
                  (-> nav
                      (zip/step-right)
                      (zip/delete-right)
                      (zip/step-left)
                      (recur))
                  
                  (type/void-block? (zip/right-element nav))
                  (-> nav
                      (zip/delete-right)
                      (zip/delete-right))
                  
                  :else
                  (-> nav
                      (zip/step-right)
                      (zip/delete-right)
                      (zip/step-left)))))))

(defn delete
  "deletes the current element
 
   (-> (parse-string \"(  #|1   2 3)\")
       (delete)
       str)
   => \"<0,3> (  |2 3)\""
  {:added "3.0"}
  [nav]
  (cond (level-empty? nav)
        (tighten-right nav)
        
        :else
        (loop [nav (-> nav
                       (position-right)
                       (zip/delete-right))]
          (let [elem (zip/right-element nav)]
            (cond (nil? elem)
                  nav
                  
                  (base/expression? elem)
                  nav
                  
                  :else
                  (recur (zip/delete-right nav)))))))

(defn backspace
  "the reverse of insert
 
   (-> (parse-string \"(0  #|1   2 3)\")
       (backspace)
       str)
   => \"<0,1> (|0 2 3)\""
  {:added "3.0"}
  [nav]
  (let [nav (-> nav
                (tighten-left)
                (delete))]
    (if-let [pnav (left nav)]
      pnav
      nav)))

(defn replace
  "replaces an element at the cursor
 
   (-> (parse-string \"(0  #|1   2 3)\")
       (position-right)
       (replace :a)
       str)
   => \"<0,4> (0  |:a   2 3)\""
  {:added "3.0"}
  [nav data]
  (cond (level-empty? nav)
        (throw (ex-info "Replace failed - level empty." {:nav nav}))
        
        :else
        (let [nav (position-right nav)]
          (cond (base/expression? (zip/right-element nav))
                (zip/replace-right nav data)
                
                :else
                (throw (ex-info "Replace failed - no element" {:nav nav}))))))

(defn update-children
  "replaces the current children
 
   (-> (update-children (parse-string \"[1 2 3]\")
                        [(construct/block 4)
                         (construct/space)
                         (construct/block 5)])
       str)
   => \"<0,0> |[4 5]\""
  {:added "3.0"}
  [nav children]
  (let [elem (zip/right-element nav)
        nelem (base/replace-children elem children)]
    (zip/replace-right nav nelem)))

(defn line-info
  "returns the line info for the current block
   
   (line-info (parse-string \"[1 \\n  2 3]\"))
   => {:row 1, :col 1, :end-row 2, :end-col 7}"
  {:added "3.0"}
  [nav]
  (let [block (zip/get nav)
        [y x] (:position nav)
        height (base/block-height block)
        width  (base/block-width block)
        [row col] [(inc y) (inc x)]]
    {:row row
     :col col
     :end-row (+ row height)
     :end-col (if (zero? height)
                (+ row width)
                (inc width))}))

(defn copy-nav-var
  "helper function for importing vars
 
   (copy-nav-var *ns* '-tag- #'hara.code.block.base/block-tag)
   => #'-tag-"
  {:added "3.0"}
  [_ name source]
  (let [var (eval `(defn ~name
                     ([~'zip]
                      (~name ~'zip :right))
                     ([~'zip ~'step]
                      (if-let [elem# (zip/get ~'zip)]
                        ((deref ~source) elem#)))))]
    (alter-meta! var
                 merge
                 (-> (meta source)
                     (dissoc :name :arglists :doc)))
    var))

(module/include {:fn copy-nav-var}
           (hara.code.block.base block?
                                 expression?
                                 (type       < block-type)
                                 (tag        < block-tag) 
                                 (string     < block-string)
                                 (length     < block-length)
                                 (width      < block-width)
                                 (height     < block-height)
                                 (prefixed   < block-prefixed)
                                 (suffixed   < block-suffixed)
                                 (verify     < block-verify)
                                 (value      < block-value)
                                 (value-string < block-value-string)
                                 (children   < block-children)
                                 (info       < block-info))
           (hara.code.block.type (void?      < void-block?)
                                 (space?     < space-block?)
                                 (linebreak? < linebreak-block?)
                                 (linespace? < linespace-block?)
                                 (eof?       < eof-block?)
                                 (comment?   < comment-block?)
                                 (token?     < token-block?)
                                 (container? < container-block?)
                                 (modifier?  < modifier-block?)))
