(ns hara.code.framework.cache
  (:require [hara.io.file :as fs]
            [hara.data.base.map :as map]
            [hara.code.framework.common :as common]
            [hara.code.block :as block])
  (:refer-clojure :exclude [update]))

(defonce +cache-dir+ ".hara/code/framework")

(defonce +cache-suffix+ ".cache")

(defonce +init-cache+ (do (fs/create-directory +cache-dir+)
                          true))

(defonce +current (atom {}))

(defn prepare-out
  "prepares code block for output
 
   (prepare-out
        {'ns {'var
              {:test {:code [(block/block '(1 2 3))]}}}})
   => '{ns {var {:test {:code \"(1 2 3)\"}}}}"
  {:added "3.0"}
  [data]
  (map/map-vals
   (fn [ns-entry]
     (map/map-vals
      (fn [entry]
        (if (-> entry :test :code)
          (update-in entry
                     [:test :code]
                     (comp block/string block/root))
          entry))
      ns-entry))
   data))

(defn prepare-in
  "prepares input string to code blocks
 
   (-> (prepare-in
        '{ns {var
              {:test {:code \"(+ 1 2 3)\"}}}})
       (get-in '[ns var :test :code])
       (first)
       (block/value))
   => '(+ 1 2 3)"
  {:added "3.0"}
  [data]
  (map/map-vals
   (fn [ns-entry]
     (map/map-vals
      (fn [entry]
        (if (-> entry :test :code)
          (update-in entry
                     [:test :code]
                     (comp vec block/children block/parse-root))
          entry))
      ns-entry))
   data))

(defn fetch
  "reads a the file or gets it from cache"
  {:added "3.0"}
  ([ns path]
   (fetch ns path +current +cache-dir+))
  ([ns path current cache-dir]
   (let [path-time (-> (fs/path path)
                       (fs/attributes)
                       (:last-modified-time))
         entry (get @current ns)]
     (if (and entry
              (< path-time
                 (:time entry)))
       (:data entry)))))

(defn update
  "updates values to the cache"
  {:added "3.0"}
  ([ns data]
   (update ns data +current +cache-dir+ +cache-suffix+))
  ([ns data current cache-dir cache-suffix]
   (let [out-path (fs/path cache-dir (str ns cache-suffix))
         _ (spit out-path (prepare-out data))
         out-time (-> out-path
                      (fs/attributes)
                      (:last-modified-time))]
     (swap! current assoc ns {:time out-time
                              :data data}))))

(defn init
  "initialises the cache with results from .code.cache"
  {:added "3.0"}
  ([] (init +current +cache-dir+))
  ([current cache-dir]
   (if (fs/exists? cache-dir)
     (fs/create-directory cache-dir))
   (let [files (fs/select cache-dir {:include [fs/file?]})]
     (mapv (fn [file]
             (let [time  (:last-modified-time (fs/attributes file))
                   data  (try (common/entry (prepare-in (read-string (slurp file))))
                              (catch Throwable t))
                   ns    (first (keys data))]
               (when data
                 (swap! current assoc ns {:time time
                                          :data data})
                 ns)))
           files))))

(defn purge
  "clears the cache"
  {:added "3.0"}
  ([]
   (purge +current))
  ([current]
   (reset! +current {})))

(defonce +status+ (future (init)))
