(ns hara.code.unit.scaffold
  (:require [clojure.set :as set]
            [hara.string :as string]
            [hara.io.file :as fs]
            [hara.io.project :as project]
            [hara.code.navigate :as nav]
            [hara.code.block :as block]))

(def ^:dynamic *ns-form* 'hara.test)

(def ^:dynamic *ns-form-fn*
  (fn [source-ns test-ns test-form]
    (format "(ns %s\n  (:use %s)\n  (:require [%s :refer :all]))"
            (str test-ns)
            (str test-form)
            (str source-ns))))

(defn test-fact-form
  "creates a fact form for the namespace
 
   (test-fact-form 'lucid 'hello \"1.1\")
   => \"^{:refer lucid/hello :added \\\"1.1\\\"}\\n(fact \\\"TODO\\\")\""
  {:added "1.2"}
  [ns var version]
  (-> [(format "^{:refer %s/%s :added \"%s\"}"
                ns var version)
      (format "(fact \"TODO\")")]
      (string/joinl "\n")))

(defn new-filename
  "creates a new file based on test namespace
 
   (new-filename 'lucid.hello-test (project/project) false)
   => (str (fs/path \"test/lucid/hello_test.clj\"))"
  {:added "3.0"}
  [test-ns project write]
  (let [extension (project/file-suffix)
        path (format "%s/%s/%s"
                     (:root project)
                     (first (:test-paths project))
                     (-> test-ns str ^String (munge) (.replaceAll "\\." "/") (str extension)))]
    (when write
      (fs/create-directory (fs/parent path)))
    path))

(defn scaffold-new
  "creates a completely new scaffold"
  {:added "1.2"}
  [source-ns test-ns vars version]
  (->> (map #(test-fact-form source-ns % version) vars)
       (cons (*ns-form-fn* source-ns test-ns *ns-form*))
       (string/join "\n\n")))

(defn scaffold-append
  "creates a scaffold for an already existing file"
  {:added "1.2"}
  [original source-ns new-vars  version]
  (if new-vars
    (->> new-vars
         (map #(test-fact-form source-ns % version))
         (string/join "\n\n")
         (str original "\n\n"))
    original))

(defn scaffold-arrange
  "arranges tests to match the order of functions in source file"
  {:added "3.0"}
  [original source-vars]
  (let [source-lu   (set source-vars)
        all-nodes   (->> (nav/parse-root original)
                         (nav/down)
                         (iterate nav/right)
                         (take-while identity)
                         (map nav/block))
        key-var    #(-> % :children first block/value :refer name symbol source-lu)
        is-ns?     #(-> % block/value first (= 'ns))
        is-var?    #(and (-> % block/tag (= :meta))
                         (key-var %))
        ns-nodes    (->> all-nodes (filter is-ns?))
        var-table   (->> all-nodes
                         (filter is-var?)
                         (map (juxt key-var identity))
                         (into {}))
        var-nodes   (keep var-table source-vars)
        other-nodes (->> all-nodes (remove is-ns?) (remove is-var?))]
    (->> (concat ns-nodes var-nodes other-nodes)
         (map block/string)
         (string/join "\n\n"))))
