(ns hara.code.manage.ns-rename
  (:require [hara.code.framework :as base]
            [hara.string.base.ansi :as ansi]
            [hara.print :as print]
            [hara.lib.diff :as diff]
            [hara.io.file :as fs]
            [hara.io.project :as project]))

(defn- subs-trim
  ([s end]
   (subs-trim s 0 end))
  ([s start end]
   (subs s start (- (count s) end))))

(defn- remove-suffix
  [^String s]
  (if (< 0 (.indexOf s "."))
    (subs s 0 (.lastIndexOf s "."))
    s))

(defn- dot-pattern
  [^String s]
  (.replaceAll s "\\." "\\\\."))

(defn- replace-fn
  [^String old ^String new]
  (fn [^String original]
    (.replaceAll original
                 (dot-pattern (str old))
                 (str new))))

(defn move-list
  "create list of files that need to be moved
 
   (project/in-context
    (move-list '[hara.module.classloader
                hara.module.classloader]
               {:print {:function true}}))"
  {:added "3.0"}
  ([[old new] {:keys [write print] :as params} _ {:keys [source-paths test-paths root]}]
   (let [old (fs/ns->file old)
         new (fs/ns->file new)
         list-fn (fn [path] (->> (fs/select (fs/path root path)
                                            {:include [old]})
                                 (map str)
                                 (map (juxt identity #(.replace ^String % old new)))))
         mlist (mapcat list-fn (concat source-paths test-paths))
         simulate (not write)]
     (if (:function print)
       (print/print-title (format "MOVES (%s)" (count mlist))))
     (mapv (fn [i [old new]]
             (if (:function print)
               (println
                (ansi/blue (format "%4s" (inc i)))
                (ansi/bold (str (fs/relativize root new)))
                '<=
                (str (fs/relativize root old))))
             (fs/move old new {:simulate simulate}))
           (range (count mlist))
           mlist)
     mlist)))

(defn change-list
  "create list of changes made to existing files
 
   (project/in-context
    (change-list '[hara.module.classloader
                  hara.module.classloader]
                 {:print {:function true
                          :progress true}}))"
  {:added "3.0"}
  [[old new] {:keys [write print] :as params} lookup project]
  (let [transform (replace-fn old new)
        changes (->> (sort (keys lookup))
                     (keep (fn [id]
                             (let [change (base/transform-code id {:transform transform
                                                                   :print {:function false}
                                                                   :write write
                                                                   :full true}
                                                               lookup project)]
                               (when (seq (:deltas change))
                                 [id change])))))]
    (when (:function print)
      (print/print-title (format "CHANGES (%s" (count changes)))
      (clojure.core/print "\n")
      (doseq [[id change] changes]
        (clojure.core/print (format (ansi/style (str (:path change)) #{:blue :bold}))
                            "\n")
        (clojure.core/print (diff/->string (:deltas change))
                            "\n")))
    changes))

(defn ns-rename
  "moves both src and test files of the package
 
   (project/in-context
    (refactor '[hara.lib.aether hara.module.deps]
             {:print {:function true}}))"
  {:added "3.0"}
  [[old new] {:keys [write print] :as params} lookup project]
  (let [changes (change-list [old new] params lookup project)
        moves   (move-list [old new] params lookup project)]
    {:changes changes
     :moves moves}))

(comment
  
  (project/in-context (refactor '[hara.code.manage.refactor
                                  hara.code.manage.refactoring]
                                {:print {:function true}}))
  
  (project/in-context (refactor '[hara.module.artifact.bitstream
                                  hara.module.artifact.bytes]
                                {:print {:function true}})))
