(ns hara.code
  (:require [hara.code.framework :as base]
            ;;[hara.code.format.lint :as lint]
            ;;[hara.code.format.ns-form :as ns-form]
            ;;[hara.code.format.var :as var]
            [hara.code.manage.ns-rename :as ns-rename]
            [hara.code.manage.ns-format :as ns-format]
            [hara.code.manage.var :as var]
            [hara.code.unit.template :as template]
            [hara.code.unit :as unit]
            [hara.module :as module]
            [hara.core.base.result :as result]
            [hara.io.project :as project]
            [hara.function :as fn :refer [definvoke]]
            [hara.function.task :as task])
  (:refer-clojure :exclude [import]))

(defmethod task/task-defaults :code
  [_]
  template/code-default)

(defmethod task/task-defaults :code.transform
  [_]
  template/code-transform)

(defmethod task/task-defaults :code.locate
  [_]
  template/code-locate)

(module/include
 (hara.code.manage.ns-rename ns-rename))

(definvoke analyse
  "analyse either a source or test file
 
   (analyse 'hara.code)
   ;;#code{:test {hara.code [analyse .... vars]}}
   => hara.code.framework.common.Entry
 
   (analyse '#{hara.code} {:return :summary})
   ;; {:errors 0, :warnings 0, :items 1, :results 1, :total 16}
   => map?"
  {:added "3.0"}
  [:task {:template :code
          :params  {:title "ANALYSE NAMESPACE"
                    :print {:result false :summary false}
                    :sorted true}
          :main    {:fn #'base/analyse}
          :item    {:display #(->> (vals %) (mapcat keys) sort vec)}
          :result  {:ignore  nil
                    :keys    {:count #(->> (vals %) (mapcat keys) count)
                              :functions #(->> (vals %) (mapcat keys) sort vec)}
                    :columns (template/code-default-columns :functions #{:bold})}}])

(comment (hara.code/analyse ['hara.code.format] {:print {:item true :result true :summary true}}))

(definvoke vars
  "returns the list of vars in a namespace
 
   (vars 'hara.code)
   
   (vars 'hara.code {:sorted false})
   
   (vars '#{hara.code} {:return #{:errors :summary}})
   => (contains-in {:errors any
                    :summary {:errors 0
                              :warnings 0
                              :items 1
                              :results 1
                              :total number?}})"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "NAMESPACE VARS"
                   :sorted true
                   :print {:result false :summary false}}
          :main {:fn #'base/vars}
          :item {:display identity}
          :result {:columns (template/code-default-columns :data #{:bold})}}])
  

(comment (hara.code/vars ['hara.code.format] {:print {:item true :result true :summary true}}))

(definvoke docstrings
  "returns docstrings
 
   (docstrings '#{hara.code.unit} {:return :results})
   ;;{:errors 0, :warnings 0, :items 1, :results 1, :total 14}
   => map?"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "VAR DOCSTRINGS"
                   :print {:result false :summary false}}
          :main   {:fn #'base/docstrings}
          :item   {:display (comp (template/empty-status :info :none) vec keys)}
          :result {:keys  {:count (comp count keys)
                           :functions (comp vec keys)}
                   :columns (template/code-default-columns :functions #{:bold})}}])


(comment (hara.code/docstrings ['hara.code.format] {:print {:item true :result true :summary true}}))

(definvoke transform-code
  "helper function for any arbitrary transformation of text
 
   (transform-code {:transform #(str % \"\\n\\n\\n\\nhello world\")})
   ;; {:deletes 0, :inserts 5, :changed [arrange], :updated false}
   => map?
   
   (transform-code '#{hara.code.unit}
                   {:print {:summary true :result true :item true :function true}
                    :transform #(str % \"\\n\\n\\n\\nhello world\")
                    :full true})"
  {:added "3.0"}
  [:task {:template :code.transform
          :params {:title "TRANSFORM CODE"}
          :main   {:fn #'base/transform-code}
          :result template/base-transform-result}])

(comment
  (transform-code ['hara.code.framework]
                  {:transform #(str % "\n\n\n\nhello world")
                   :print {:summary true :result true :item true :function true}}))

(definvoke import
  "import docstrings from tests
 
   (import {:write false})
 
   (import {:full true
            :write false
            :print {:function false}})
 
   (import '[hara.code.unit]
           {:print {:summary true :result true :item true}
            :write false})"
  {:added "3.0"}
  [:task {:template :code.transform
          :main   {:fn #'unit/import}
          :params {:title "IMPORT DOCSTRINGS"
                   :write true}
          :item   {:list template/source-namespaces}
          :result (template/code-transform-result :changed)}])

(comment (hara.code/import ['hara.code.framework] {:print {:item true :result true :summary true}}))

(definvoke purge
  "removes docstrings from source code
 
   (purge {:write false})
   
   (purge {:full true :write false})
 
   (purge '[hara.core.unit] {:return :summary :write false})
   ;;{:items 38, :results 32, :deletes 1272, :total 169}
   => map?"
  {:added "3.0"}
  [:task {:template :code.transform
          :main   {:fn #'unit/purge}
          :params {:title "PURGE DOCSTRINGS"}
          :item {:list template/source-namespaces}
          :result (assoc (template/code-transform-result :changed)
                         :columns (template/code-transform-columns #{:bold :red}))}])
  

(comment (hara.code/purge ['hara.code] {:print {:item true :result true :summary true}}))

(definvoke missing
  "checks for functions with missing tests
 
   (missing)
   
   (missing '[hara.core] {:print {:result false :summary false}
                                  :return :all})"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "MISSING TESTS"}
          :main {:fn #'unit/missing}
          :item {:list template/source-namespaces}
          :result {:columns (template/code-default-columns :data #{:green})}}])

(comment (hara.code/missing ['hara.code] {:print {:item true :result true :summary true}}))

(definvoke todos
  "checks for tests with `TODO` as docstring
 
   (todos)
 
   (todos '[hara.core] {:print {:result false :summary false}
                          :return :all})"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "TODO TESTS"}
          :main {:fn #'unit/todos}
          :item {:list template/test-namespaces}
          :result {:columns (template/code-default-columns #{:green})}}])

(comment (hara.code/todos ['hara.code] {:print {:item true :result true :summary true}}))

(definvoke incomplete
  "both functions missing tests or tests with todos
 
   (incomplete)
   
   (incomplete '[hara.code] {:print {:item true}})"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "INCOMPLETE TESTS"
                   :print {:item true}}
          :main {:fn #'unit/incomplete}
          :item {:list template/source-namespaces}
          :result {:columns (template/code-default-columns #{:bold :green})}}])

(comment (hara.code/incomplete ['hara.code] {:print {:item true :result true :summary true}}))

(definvoke orphaned
  "tests without corresponding source code
 
   (orphaned)
 
   (orphaned '[hara.code] {:print {:item true}})"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "ORPHANED TESTS"}
          :main {:fn #'unit/orphaned}
          :item {:list template/test-namespaces}
          :result {:columns (template/code-default-columns #{:bold :blue})}}])
  
(comment (hara.code/orphaned ['hara.code] {:print {:item true :result true :summary true}}))

(definvoke scaffold
  "creates a scaffold for a new or existing set of tests
 
   (scaffold {:write false})
 
   (scaffold '[hara.code] {:print {:item true}
                           :write false})"
  {:added "3.0"}
  [:task {:template :code.transform
          :params {:title "CREATE TEST SCAFFOLD"
                   :write true}
          :main {:fn #'unit/scaffold}
          :item {:list template/source-namespaces
                 :display (template/empty-result :new :info :no-new)}
          :result (template/code-transform-result :new)}])

(comment (hara.code/scaffold ['hara.code.framework] {:print {:function true :item true :result true :summary true}}))

(defn- compare-status [arr]
  (cond (empty? arr)
        (result/result {:status :info
                        :data :ok})

        :else
        (result/result {:status :warn
                        :data :not-in-order})))

(defn- compare-columns
  [orig compare]
  [{:key    :key
    :align  :left}
   {:key    :count
    :length 8
    :align  :center
    :color  #{:bold}}
   {:key    :test
    :align  :left
    :length 60
    :color  orig}
   {:key    :source
    :align  :left
    :length 60
    :color  compare}])

(definvoke in-order?
  "checks if tests are in order
 
   (in-order?)
   
   (in-order? '[hara.code] {:print {:item true}})"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "TESTS IN ORDER"}
          :main {:fn #'unit/in-order?}
          :item {:list template/source-namespaces
                 :display compare-status}
          :result {:keys {:count first
                          :source second
                          :test #(nth % 2)}
                   :columns (compare-columns #{:bold :cyan} #{:cyan})}}])

(comment (hara.code/in-order? ['hara.code.framework] {:print {:function true :item true :result true :summary true}}))

(definvoke arrange
  "arranges the test corresponding to function order
 
   (arrange {:print {:function false}
             :write false})
 
   (arrange '[hara.code] {:print {:item true}
                          :write false})"
  {:added "3.0"}
  [:task {:template :code.transform
          :params {:title "ARRANGE TESTS" :write true}
          :main {:fn #'unit/arrange}
          :item {:list template/test-namespaces}
          :result (template/code-transform-result :changed)}])

(comment (hara.code/arrange ['hara.code.framework] {:print {:function true :item true :result true :summary true}}))

(definvoke locate-code
  "locates code base upon query
 
   (locate-code '[hara.code]
                {:query '[ns | {:first :import}]})"
  {:added "3.0"}
  [:task {:template :code.locate
          :params {:title "LOCATE CODE"}
          :main {:fn #'base/locate-code}}])

(comment (hara.code/locate-code '[hara.code.framework]
                                {:query ['comment]
                                 ;;:print {:function true :item true :result true :summary true}
                                 }))

(definvoke unclean
  "finds source code that has top-level comments
 
   (unclean 'hara.code)
 
   (unclean '[hara])"
  {:added "3.0"}
  [:task {:template :code.locate
          :params {:title "SOURCE CODE WITH COMMENTS"
                   :query '[comment]}
          :main {:fn #'base/locate-code}
          :result {:columns (template/code-default-columns :source #{:red :bold})}}])

(comment
  (hara.code/unclean '[hara.code.framework] {:print {:function true :item true :result true :summary true}}))

(definvoke unchecked
  "returns tests without `=>` checks
 
   (unchecked)
 
   (unchecked '[hara.code])
   "
  {:added "3.0"}
  [:task {:template :code
          :params {:title "FACT HAS NO `=>` FORMS"}
          :main {:fn #'unit/unchecked}
          :item {:list template/source-namespaces}
          :result {:columns (template/code-default-columns #{:warn :bold})}}])

(definvoke commented
  "returns tests that are in comment blocks
 
   (commented)
   
   (commented '[hara.code])"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "REFERENCED VAR IN COMMENT FORM"}
          :main {:fn #'unit/commented}
          :item {:list template/source-namespaces}
          :result {:columns (template/code-default-columns #{:warn :bold})}}])
  
(definvoke pedantic
  "returns tests that may be improved
 
   (pedantic)
 
   (pedantic '[hara.code])"
  {:added "3.0"}
  [:task {:template :code
          :params {:title "FUNCTIONS THAT COULD BE IMPROVED UPON"}
          :main {:fn #'unit/pedantic}
          :item {:list template/source-namespaces}
          :result {:columns (template/code-default-columns #{:warn :bold})}}])

(definvoke refactor-code
  "refactors code based on given `:edits`
 
   (refactor-code '[hara.code]
                  {:edits []})"
  {:added "3.0"}
  [:task {:template :code.transform
          :params {:title "REPLACE VAR USAGES"
                   :print {:function true}}
          :main   {:fn #'base/refactor-code}
          :result template/base-transform-result}])

(definvoke ns-format
  "refactors code based on given `:edits`
 
   (refactor-code '[hara.code]
                  {:edits []})"
  {:added "3.0"}
  [:task {:template :code.transform
          :params {:title "FORMAT NS FORMS"
                   :print {:function true}}
          :main   {:fn #'ns-format/ns-format}
          :result template/base-transform-result}])

(definvoke find-usages
    "find usages of a var
 
   (find-usages '[hara.code]
                {:var 'hara.code.framework/analyse})"
    {:added "3.0"}
    [:task {:template :code.locate
            :params {:title "ALL VAR USAGES"
                     :highlight? true}
            :main {:fn #'var/find-usages}}])

(comment (find-usages '[hara.code] {:print {:item true} :var 'hara.code.framework/analyse}))

(comment
  

  (definvoke replace-usages
    "replace usages of a var
 
   (replace-usages '[hara.code]
                   {:var 'hara.code.framework/analyse
                   :new 'analyse-changed})"
    {:added "3.0"}
    [:task {:template :code.transform
            :params {:title "REPLACE VAR USAGES"
                     :print {:item false
                             :function true}}
            :main   {:fn #'var/replace-usages}
            :result (template/code-transform-result :changed)}])

  (definvoke replace-refers
    "TODO"
    {:added "3.0"}
    [:task {:template :code.transform
            :params {:title "REPLACE REFERS"
                     :print {:item false
                             :function true}}
            :main   {:fn #'var/replace-refers}
            :result (template/code-transform-result :changed)}])

  (definvoke replace-errors
    "TODO"
    {:added "3.0"}
    [:task {:template :code.transform
            :params {:title "REPLACE ERRORS"
                     :print {:item false
                             :function true}}
            :main   {:fn #'var/replace-errors}
            :result (template/code-transform-result :changed)}])

  (definvoke list-ns-unused
    "TODO"
    {:added "3.0"}
    [:task {:template :code
            :params {:title "LIST UNUSED NS ENTRIES"
                     :print {:item true}}
            :item {:list template/source-namespaces}
            :main   {:fn #'var/list-ns-unused}}])

  (definvoke remove-ns-unused
    "TODO"
    {:added "3.0"}
    [:task {:template :code.transform
            :params {
                     :title "REMOVE UNUSED NS ENTRIES"
                     :print {:function true}}
            :main   {:fn #'var/remove-ns-unused}
            :result template/base-transform-result}])

  (definvoke rename-ns-abbrevs
    "TODO"
    {:added "3.0"}
    [:task {:template :code.transform
            :params {:title "RENAME NS ABBREVS"
                     :print {:function true}}
            :main {:fn #'var/rename-ns-abbrevs}
            :result template/base-transform-result}])

  (definvoke refactor-ns-forms
    "refactors and reorganises ns forms
 
   (refactor-ns-forms '[hara.code])"
    {:added "3.0"}
    [:task {:template :code.transform
            :params {:title "TRANSFORMING NS FORMS"
                     :print {:function true}}
            :main {:fn #'ns-form/refactor}
            :result template/base-transform-result}])

  (comment
    (hara.code/refactor-ns-forms '[hara.code] {:write true}))

  (definvoke lint
    [:task {:template :code.transform
            :params {:title "LINTING CODE"
                     :print {:function true}}
            :main   {:fn #'lint/lint}
            :result template/base-transform-result}])

  (comment
    (hara.code/lint 'hara.code {:write true}))

  (definvoke line-limit
    [:task {:template :code
            :params {:title "LINES EXCEEDING LIMIT"
                     :print {:function true}}
            :main   {:fn #'lint/line-limit}}])

  (comment
    (time (analyse 'hara.code.framework-test))
    (time (def a (analyse 'hara.code.framework-test)))
    (time (def a (analyse 'hara.code)))
    (hara.code.framework.cache/purge)
    (./reset '[hara.code])
    (./reset '[hara.code])
    (./incomplete '[hara.code])


    
    (vars 'hara.code)
    (vars '[thing] {:print {:item true}})
    (./import)
    (hara.code/line-limit ['hara] {:length 110}))
  )
  
