(ns active.clojure.simple-types)

(defprotocol IUnapply
  (-unapply [this v] "Return nil if v does not match, or a vector of the values this was composed from."))

(defprotocol IType
  (-is-of? [this v] "Returns if v is of this type."))

(defn type? [v]
  (satisfies? IType v))

(defn is-of? [t v]
  (or (and (satisfies? IType t) (-is-of? t v))
      (and (ifn? t) (t v))))

(defrecord ^:private RecordType [name fields])

(defn record-type [name fields]
  (RecordType. name fields))

(defn record-type? [v]
  (instance? RecordType v))

;; TODO: record->map, map->record
;; TODO: something special for singleton types? (empty records)

(declare record=)
(declare record-hash)

#?(:clj (deftype ^:private Record [type values]
                 clojure.lang.IHashEq
                 (hasheq [this] (record-hash this))
                 (hashCode [this] (record-hash this))
                 (equals [this other] (record= this other))
                 ;; TODO:
                 #_clojure.lang.IObj
                 #_(meta [this] )
                 #_(withMeta)
                 ;; ILookup ?
                 ;; TODO: printing?!
                 java.io.Serializable))

#?(:cljs (deftype ^:private Record [type values]
                  IEquiv
                  (-equiv [this other] (record= this other))
                  ICloneable
                  (-clone [this] (Record. type values))
                  IHash
                  (-hash [this] (record-hash this))
                  ;; TODO: IMeta, IWithMeta
                  IPrintWithWriter
                  (-pr-writer [this writer opts]
                    (cljs.core/pr-sequential-writer writer
                                                    (fn [[field v]]
                                                      (cljs.core/-write writer ":")
                                                      (cljs.core/-write writer (str field))
                                                      (cljs.core/-write writer " ")
                                                      (cljs.core/pr-writer v writer opts))
                                                    (str "#" (.-name type) "{")
                                                    ", "
                                                    "}"
                                                    opts
                                                    (map vector (.-fields type) values)))))

(defn- record-hash [r]
  ;; TODO: something not producing garbage??
  (hash [::record (.-type r) (.-values r)]))

(defn- record= [r other]
  (and (instance? Record other)
       (= (.-type r) (.-type other))
       (= (.-values r) (.-values other))))

(defn is-of-record-type? [t v]
  (assert (record-type? t))
  (and (instance? Record v)
       (= (.-type v) t)))

(extend-type RecordType
  IType
  (-is-of? [this v] (is-of-record-type? this v))
  IUnapply
  (-unapply [this v]
    (when (and (instance? Record v) (is-of-record-type? this v))
      (.-values v))))

(defmacro define-record-type [name & args]
  #_[name ctor-form pred field-specs]
  (let [[ctor-form pred field-specs] (if (map? (first args)) ;; ignore options. TODO: remove.
                                       (rest args)
                                       args)]
  (when-not (list? ctor-form)
    (throw (ex-info "Constructor form must be a list." {:record-type name})))
  (when-not (symbol? pred)
    (throw (ex-info "Predicate name must be a symbol." {:record-type name})))
  (when-not (vector? field-specs)
    (throw (ex-info "Field specifications must be a vector." {:record-type name})))
  (when-not (even? (count field-specs))
    (throw (ex-info "Field specifications must contains an even number of forms." {:record-type name})))
  (when-not (= (count (rest ctor-form))
               (/ (count field-specs) 2))
    (throw (ex-info "Number of constructor arguments must equal the number of field specifications." {:record-type name})))
  (when-not (every? symbol? (rest ctor-form))
    (throw (ex-info "Constructor arguments must be specified as symbols." {:record-type name})))
  (let [specs (partition 2 field-specs)]
    (when-not (= (rest ctor-form)
                 (map first specs))
      (throw (ex-info "Constructor arguments must match the fields specified." {:record-type name})))

    `(do
       (def ~name (record-type '~(symbol (str *ns*) (str name))
                               ~(mapv (fn [[n _]]
                                        `'~n)
                                      specs)))

       (defn ~(first ctor-form) [~@(rest ctor-form)]
         (Record. ~name [~@(rest ctor-form)]))

       (defn ~pred [v#]
         (is-of-record-type? ~name v#))

       ~@(map-indexed (fn [idx acc]
                        `(defn ~acc
                           ([v#]
                            (assert (~pred v#) v#)
                            (nth (.-values v#) ~idx))
                           ([v# vv#]
                            (assert (~pred v#) v#)
                            (Record. (.-type v#) (assoc (.-values v#) ~idx vv#)))))
                      (map second specs))))))

(defrecord ^:private SumType [name types])

(defn sum-type [name types]
  (SumType. name types))

(defn sum-type? [v]
  (instance? SumType v))

(defn is-of-sum-type? [t v]
  (assert (sum-type? t))
  (some #(is-of? % v) (.-types t)))

(extend-type SumType
  IType
  (-is-of? [this v] (is-of-sum-type? this v))
  IUnapply
  (-unapply [this v]
    (when (is-of-sum-type? this v)
      [v])))

(defmacro define-sum-type [name pred types]
  ;; TODO: allow any predicate as the type.
  (let [vv (gensym)]
    `(do
       (def ~name (sum-type '~(symbol (str *ns*) (str name))
                            ~types))
     
       (defn ~pred [~vv]
         (or ~@(map (fn [t] `(is-of? ~t ~vv))
                    types))))))

(defrecord ^:private EnumType [name value-set])

(defn enum-type [name values]
  (assert (set? values))
  (EnumType. name values))

(defn enum-type? [v]
  (instance? EnumType v))

(defn is-of-enum-type? [t v]
  (assert (enum-type? t))
  (contains? (.-value-set t) v))

(extend-type EnumType
  IType
  (-is-of? [this v] (is-of-enum-type? this v))
  IUnapply
  (-unapply [this v]
    (when (is-of-enum-type? this v)
      [v])))

(defmacro define-enum-type [name pred values]
  `(do
     (def ~name (enum-type '~(symbol (str *ns*) (str name))
                           ~values))
     
     (defn ~pred [v#]
       (is-of-enum-type? ~name v#))))

(defn- unapply-pred [v pred args]
  (let [items (gensym)]
    `(if (satisfies? IUnapply ~pred)
       (let [~items (-unapply ~pred ~v)]
         (and (some? ~items)
              ~@(map-indexed (fn [idx x]
                               (if (list? x)
                                 (unapply-pred `(nth ~items ~idx) (first x) (rest x))
                                 true))
                             args)))
       (do (assert (= 1 ~(count args)))
           (~pred ~v)))))

(defn- unapply-body [v pred args body]
  (let [reps (repeatedly (count args) gensym)]
    `(let [[~@(map (fn [arg rep]
                     (if (list? arg)
                       rep
                       arg))
                   args reps)]
           (if (satisfies? IUnapply ~pred)
             (-unapply ~pred ~v)
             [~v])]
       ~(reduce (fn [body [arg rep]]
                  (if (list? arg)
                    (unapply-body rep (first arg) (rest arg) body)
                    body))
                body
                (map vector args reps)))))

(defn- unapply-cond [v form body]
  (cond
    (list? form)
    (let [pred (first form)
          args (rest form)]
      [(unapply-pred v pred args)
       (unapply-body v pred args body)])

    :else
    [true
     `(let [~form ~v] ~body)]))

(defmacro match
  "A conditional special form over values, with special
  deconstructuring support for type implementing IUnapply"
  [v & clauses]
  (when-not (even? (count clauses))
    (ex-info "Invalid clauses" {}))
  (let [preds (map first (partition 2 clauses))
        bodys (map second (partition 2 clauses))

        has-else? (some #(= :else %) preds)

        vv (gensym)]
    (when-not (or (not-any? #(= :else %)
                            preds)
                  (= :else (last preds)))
      (ex-info "Invalid clauses" {}))

    ;; TODO: use condp with :>> clauses?
    `(do
       (let [~vv ~v]
         (cond
           ~@(mapcat (fn [pred body]
                       (cond
                         (= :else pred)
                         [:else body]
                         
                         (list? pred)
                         (unapply-cond vv pred body)
                         
                         :else
                         [`(~pred ~vv) body]))
                     preds bodys)

           ~@(when-not has-else?
               [:else `(throw (ex-info "Non-exhausive match." {:value ~vv}))]))))))
