(ns com.beardandcode.forms.render
  (:require [clojure.string :as s]
            [clojure.set :refer [difference]]))

(defn- pick-title [name details]
  (let [title (details "title")]
    (if (nil? title)
      (s/capitalize (s/replace name #"[-_]" " "))
      title)))

(defn- password? [details]
  (= (details "format") "password"))

(defn- sort-properties [schema-properties ordered-names-raw]
  (let [all-names (set (keys schema-properties))
        ordered-names (filter all-names ordered-names-raw)
        unordered-names (difference all-names ordered-names)
        pick #(vector % (schema-properties %))]
    (concat (map pick ordered-names)
                    (map pick unordered-names))))


(defmulti render-property (fn [_ details _ _ _ _] (details "type")))


(defn schema
  ([schema-map enum-fns values errors] (schema schema-map enum-fns values errors []))
  ([schema-map enum-fns values errors prefixes]
   (render-property prefixes schema-map false enum-fns values errors)))


(defn- find-value [values path]
  (get-in values path))

(defn- find-errors [errors path]
  (errors (str "/" (clojure.string/join "/" path))))

(defn- find-enum-fn [enum-fns path]
  (get-in enum-fns path))

(defn- as-id [path] (clojure.string/join "_" path))
(defn- as-name [path] (last path))

(defn- assoc-required [m required?]
  (if required? (assoc m :required "required") m))
(defn- assoc-checked [m checked?]
  (if checked? (assoc m :checked "checked") m))

(defn error-list [errors]
  (map #(vector :p {:class "error"} %) errors))

(defn- wrap-form-item [path details required? errors form-item]
  (let [prop-errors (find-errors errors path)
        id (as-id path)
        name (as-name path)
        title (pick-title name details)
        body (concat (if (details "description") (list [:p (details "description")]) '())
             (error-list prop-errors)
             (list form-item))
        classes [(if (> (count prop-errors) 0) "error" "")
                 (if required? "required" "")]]
    (if (empty? title) body [:label {:class (s/join " " classes) :id id} title body])))

(defmethod render-property "string" [path details required? enum-fns values errors]
  (wrap-form-item path details required? errors
                  (let [id (as-id path)
                        existing-value (find-value values path)]
                    (if-let [enum-fn (find-enum-fn enum-fns path)]
                      [:select (-> {:name id}
                                   (assoc-required required?)) [:option]
                       (for [entry (enum-fn)
                             :let [value (if (sequential? entry) (first entry) entry)
                                   label (if (sequential? entry) (second entry) entry)]]
                         [:option {:value value :selected (= value existing-value)} label])]
                      [:input (-> {:type (if (password? details) "password" "text")
                                   :name id
                                   :value (if (password? details) nil existing-value)}
                                  (assoc-required required?))]))))

(defmethod render-property "boolean" [path details required? _ values errors]
  (wrap-form-item path details required? errors
                  [:input (-> {:type "checkbox"
                               :value "true"
                               :name (as-id path)}
                              (assoc-checked (find-value values path)))]))

(defmethod render-property "object" [path details required? enum-fns values errors]
  (let [required-properties (set (details "required" []))
        children (map (fn [[prop-name prop-details]]
                        (render-property (conj path prop-name) prop-details
                                         (boolean (required-properties prop-name))
                                         enum-fns values errors))
                      (sort-properties (details "properties")
                                       (details "order")))]
    (if (empty? path)
      children
      [:fieldset {:id (as-id path)}
       (concat (list [:legend (pick-title (as-name path) details)])
               (error-list (find-errors errors path))
               children)])))

(defn- num-display-fields [items details]
  (let [max-items (details "maxItems" 0)
        min-items (details "minItems" 0)
        num-items (count items)]
    (cond (> num-items max-items) num-items
          (> min-items 5) min-items
          (> max-items 0) max-items
          :else 5)))

(defmethod render-property "array" [path details required? enum-fns values errors]
  (let [item-details (assoc (details "items" {}) "title" "")
        items (find-value values path)
        item-values (if items
                      (->> items (map-indexed vector) (flatten)
                           (apply hash-map) (assoc-in values path))
                      values)]
    [:fieldset {:class (if required? "required" "") :id (as-id path)}
     (concat (list [:legend (pick-title (as-name path) details)])
             (error-list (find-errors errors path))
             (list [:ol {:class "list-entry"
                         :data-item-title ((details "items" {}) "title")
                         :data-item-type (item-details "type")
                         :data-item-path (as-id path)
                         :data-min-items (details "minItems")
                         :data-max-items (details "maxItems")}
                    (for [i (range (num-display-fields items details))
                          :let [item-path (conj path i)
                                item-errors? (> (count (find-errors errors item-path)) 0)]]
                      [:li {:class (if item-errors? "error" "")}
                       (render-property item-path item-details false enum-fns item-values errors)])]))]))

(defmethod render-property nil [path details required? _ values errors]
  (let [prop-errors (find-errors errors path)
        id (as-id path) name (as-name path)
        classes [(if (> (count prop-errors) 0) "error" "")
                 (if required? "required" "")]]
    (if-let [enum (details "enum")]
      [:fieldset {:class (s/join " " classes) :id id}
       (concat (list [:legend (pick-title name details)])
               (if (details "description") (list [:p (details "description")]) '())
               (error-list prop-errors)
               (map #(vector :label
                             [:input (-> {:type "radio" :value % :name id}
                                         (assoc-required required?)
                                         (assoc-checked (= (find-value values path) %)))]
                             (s/capitalize %)) enum))])))

(defmethod render-property :default [& args]
  (println args))
