(ns patterning.svg-tools
  #?(:clj (:require [clojure.data.xml :as xml]))
  (:require [clojure.string :as string]
            [patterning.color :refer [hex-color p-color]]
            [patterning.maths :as maths]
            [patterning.sshapes :refer [->SShape]]))

(def ^:private number-pattern
  #"[-+]?\d*\.?\d+(?:[eE][-+]?\d+)?")

(defn- parse-number [s]
  (when s
    (when-let [m (re-find number-pattern (str s))]
      (float (Double/parseDouble m)))))

(defn- parse-number-list [s]
  (map parse-number (re-seq number-pattern (str s))))

(defn- parse-points [s]
  (->> (parse-number-list s)
       (partition 2)
       (mapv vec)))

(defn- close-points [points]
  (if (and (seq points) (not= (first points) (last points)))
    (conj (vec points) (first points))
    (vec points)))

(defn- parse-color [s]
  (let [s (some-> s string/trim)]
    (cond
      (nil? s) nil
      (= "none" (string/lower-case s)) ::none
      (re-matches #"(?i)^#?[0-9a-f]{3,8}$" s) (hex-color s)
      (re-matches #"(?i)^rgb\\s*\\(.*\\)$" s)
      (let [nums (map int (parse-number-list s))]
        (when (>= (count nums) 3)
          (apply p-color (take 3 nums))))
      :else nil)))

(defn- attrs->style [attrs]
  (let [fill (parse-color (get attrs :fill))
        stroke (parse-color (get attrs :stroke))
        stroke-width (parse-number (get attrs :stroke-width))]
    (cond-> {}
      fill (assoc :fill fill)
      stroke (assoc :stroke stroke)
      stroke-width (assoc :stroke-weight stroke-width))))

(defn- merge-styles [parent child]
  (reduce-kv
   (fn [m k v]
     (if (= v ::none)
       (dissoc m k)
       (assoc m k v)))
   (or parent {})
   child))

(defn- element->sshapes [el parent-style]
  (let [attrs (:attrs el)
        style (merge-styles parent-style (attrs->style attrs))]
    (case (:tag el)
      :svg (mapcat #(element->sshapes % style) (:content el))
      :g (mapcat #(element->sshapes % style) (:content el))
      :polygon (let [points (close-points (parse-points (get attrs :points)))]
                 (if (seq points)
                   [(->SShape style points)]
                   []))
      :polyline (let [points (parse-points (get attrs :points))]
                  (if (seq points)
                    [(->SShape style points)]
                    []))
      :rect (let [x (or (parse-number (get attrs :x)) 0.0)
                  y (or (parse-number (get attrs :y)) 0.0)
                  w (or (parse-number (get attrs :width)) 0.0)
                  h (or (parse-number (get attrs :height)) 0.0)
                  points [[x y] [(+ x w) y] [(+ x w) (+ y h)] [x (+ y h)] [x y]]]
              [(->SShape style points)])
      :line (let [x1 (or (parse-number (get attrs :x1)) 0.0)
                  y1 (or (parse-number (get attrs :y1)) 0.0)
                  x2 (or (parse-number (get attrs :x2)) 0.0)
                  y2 (or (parse-number (get attrs :y2)) 0.0)]
              [(->SShape style [[x1 y1] [x2 y2]])])
      :circle (let [cx (or (parse-number (get attrs :cx)) 0.0)
                    cy (or (parse-number (get attrs :cy)) 0.0)
                    r (or (parse-number (get attrs :r)) 0.0)]
                (if (pos? r)
                  (let [points (->> (maths/clock-angles 36)
                                    (map (fn [a]
                                           (let [[dx dy] (maths/pol-to-rec [r a])]
                                             [(+ cx dx) (+ cy dy)])))
                                    vec
                                    close-points)]
                    [(->SShape style points)])
                  []))
      [])))

#?(:clj
   (defn svg->pattern
     "Parse a subset of SVG into a sequence of SShapes."
     [svg-str]
     (let [root (xml/parse-str svg-str)]
       (vec (element->sshapes root {})))))

#?(:cljs
   (defn svg->pattern
     [svg-str]
     (throw (ex-info "svg->pattern is not yet supported in ClojureScript"
                     {:input svg-str}))))
