(ns ch.codesmith.blocks.postgres.dev
  (:require [babashka.process :as ps]
            [ch.codesmith.blocks.core :as cb]
            [ch.codesmith.blocks.jdbc :as cbjdbc]
            [ch.codesmith.blocks.postgres :as cbp]
            [clojure.string :as str]
            [next.jdbc :as jdbc]
            [next.jdbc.protocols :as p])
  (:import (io.zonky.test.db.postgres.embedded DatabasePreparer EmbeddedPostgres PreparedDbProvider)
           (javax.sql DataSource)))

(defmulti stop-embedded-postgres
  {:arglists '([port])}
  (fn [_]
    (System/getProperty "os.name")))

(defn shell-str [command]
  (str/trim (:out (ps/shell {:out :string} command))))

(defn macos-pids-on-port [port]
  (map #(Integer/parseInt %) (str/split-lines (shell-str (str "lsof -ti tcp:" port)))))

(defn macos-ppid [pid]
  (Integer/parseInt (shell-str (str "ps -o ppid= " pid))))

(defn macos-command [pid]
  (shell-str (str "ps -o command= " pid)))

(defn macos-stop-command [^String postgres-command]
  (second (re-matches
            #"(.*pg_ctl stop -D \S*).*"
            (.replace postgres-command
              "/bin/postgres"
              "/bin/pg_ctl stop"))))

(defmethod stop-embedded-postgres "Mac OS X"
  [port]
  (try (ps/shell (->> port
                   macos-pids-on-port
                   (some #(and (= 1 (macos-ppid %)) %))
                   macos-command
                   macos-stop-command))
       (catch Exception _))
  nil)

(defrecord Preparer [config]
  DatabasePreparer
  (prepare [_ ds]
    (when-let [init-sql-statements (:init-sql-statements config)]
      (when (seq init-sql-statements)
        (jdbc/with-transaction [tx ds]
          (doseq [init-statement init-sql-statements]
            (jdbc/execute! tx init-statement)))))
    (cbjdbc/ensure-db-migrated! ds config)))

(defn db-preparer [config]
  (->Preparer config))

(derive ::embedded ::cbp/datasource)

(defn create-blocks-database [^EmbeddedPostgres embedded-postgres]
  (with-open [conn (.getConnection (.getPostgresDatabase embedded-postgres))]
    (jdbc/execute! conn ["create user blocks with superuser password 'password'"])
    (jdbc/execute! conn ["create database blocks owner blocks"])))

(defmethod cb/start-block!
  ::embedded
  [_ _ {:keys [force-stop? port] :as config} _]
  (when force-stop?
    (stop-embedded-postgres port))
  (let [builder                             (EmbeddedPostgres/builder)
        builder                             (if port (.setPort builder port) builder)
        ^EmbeddedPostgres embedded-postgres (.start builder)]
    (create-blocks-database embedded-postgres)
    (let [jdbc-url (.getJdbcUrl embedded-postgres "blocks" "blocks")
          config   (update config :datasource
                     #(assoc %
                        :jdbcUrl jdbc-url
                        :password "password"
                        :user "blocks"))
          ds       (cbjdbc/start-hikaricp-pool config)]
      (.prepare (db-preparer config) (p/get-datasource ds))
      {:datasource        ds
       :embedded-postgres embedded-postgres})))

(defmethod cb/resolve-block ::embedded [_ instance]
  (:datasource instance))

(defmethod cb/stop-block! ::embedded [_ {:keys [datasource ^EmbeddedPostgres embedded-postgres]}]
  (cbjdbc/halt! datasource)
  (.close embedded-postgres))

(derive ::db-provider ::cbp/datasource)

(defn db-provider [^DatabasePreparer db-preparer]
  (PreparedDbProvider/forPreparer
    (reify DatabasePreparer
      (prepare [_ ds]
        (with-open [conn (.getConnection ds)]
          (let [url (.getURL (.getMetaData conn))]
            (jdbc/execute! conn ["create user blocks with superuser password 'password'"])
            (.prepare db-preparer
              (reify DataSource
                (getConnection [ds]
                  (.getConnection ds "blocks" "password"))
                (getConnection [_ username password]
                  (jdbc/get-connection {:jdbcUrl  url
                                        :password password
                                        :user     username}))
                (getLogWriter [_])
                (setLogWriter [_ _])
                (getLoginTimeout [_] 0)
                (setLoginTimeout [_ _])))))))))

(defmethod cb/start-block! ::db-provider
  [_ _ config _]
  (let [jdbc-url (.createDatabase (db-provider (db-preparer config)))
        jdbc-url (str/replace jdbc-url "user=postgres" "user=blocks")]
    (cbjdbc/start-hikaricp-pool (merge config
                                  {:datasource {:jdbcUrl  jdbc-url
                                                :password "password"
                                                :username "blocks"}}))))

(defmethod cb/stop-block! ::db-provider
  [_ ds]
  (cbjdbc/halt! ds))
