(ns ch.codesmith.blocks
  (:refer-clojure :exclude [ref])
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [integrant.core :as ig]))

(s/def ::type keyword?)
(s/def ::config any?)
(s/def ::wrap-fn ifn?)
(s/def ::block-key (fn [value]
                     (or
                       (keyword? value)
                       (vector? value))))

(s/def ::block (s/keys :opt-un [::type ::config ::wrap-fn]))

(s/def ::name keyword?)
(s/def ::blocks (s/map-of ::block-key ::block))
(s/def ::base-config (s/keys :req-un [::name]
                       :opt-un [::blocks]))

(s/def ::profile keyword?)
(s/def ::profile-config (s/keys :req-un [::profile]
                          :opt-un [::blocks]))

(defonce
  ^{:doc      "Return a unique keyword that is derived from an ordered collection of named.
  The function will return the same keyword for the same collection.
  Cf. [[integrant.core/composite-keyword]]"
    :arglists '([nameds])}
  composite-keyword
  (memoize
    (fn [nameds]
      (let [parts  (mapv
                     (fn [named]
                       (let [name (name named)]
                         (if-let [ns (namespace named)]
                           (str ns "." name)
                           name)))
                     nameds)
            prefix (str (str/join "+" parts) "_")]
        (keyword "blocks.composite" (str (gensym prefix)))))))

(defn coerce-keyword [ref]
  (cond
    (vector? ref) (composite-keyword ref)
    (keyword? ref) ref
    :else (ex-info (str "Invalid reference " ref) {:ref ref})))

(defn ref [k]
  (ig/ref (coerce-keyword k)))

(defrecord SymbolicRef [ref])

(defn symbolic-ref
  "Creates a Symbolic Ref. It is simply a marker for a ref with symbols that can
  be used to walk a config map when using symbols as a templating mechanism."
  [ref]
  (->SymbolicRef ref))

(defn checker [spec]
  #(if (s/valid? spec %)
     %
     (throw (ex-info (str "The value " % " is not conform to its spec")
              {:schema      spec
               :value       %
               :explanation (s/explain-data spec %)}))))

(def check-base-config (checker ::base-config))
(def check-profile-config (checker ::profile-config))

;; some utils

(defmethod ig/init-key ::identity [_ value]
  value)

(defn envvar [{:keys [envvar coerce]}]
  (let [value (System/getenv envvar)]
    (if coerce
      (coerce value)
      value)))

(defmethod ig/init-key ::envvar [_ value]
  (envvar value))

(defn slurp-file [{:keys [file slurp] :or {slurp clojure.core/slurp}}]
  (slurp file))

(defmethod ig/init-key ::file [_ value]
  (slurp-file value))

;; system

;; Copied verbatim from the defunct clojure-contrib (http://bit.ly/deep-merge-with)
(defn deep-merge-with [f & maps]
  (apply
    (fn m [& maps]
      (if (every? map? maps)
        (apply merge-with m maps)
        (apply f maps)))
    maps))

(defn deep-merge [& maps]
  (apply deep-merge-with (fn [_ val] val) maps))

;; Init / Halt! / Ignition

(defn blocks [{:keys [blocks]}]
  (or blocks {}))

(defn components [{:keys [components]}]
  (or components {}))

(defn merge-with-profiles [base & profiles]
  (check-base-config base)
  (doseq [profile profiles]
    (check-profile-config profile))
  {:name       (:name base)
   :profiles   (mapv :profile profiles)
   :blocks     (apply deep-merge (blocks base) (map blocks profiles))
   :components (apply deep-merge (components base) (map components profiles))})

(defn halt! [{:keys [system]}]
  (when system (ig/halt! system)))

(defn derivation [derivations key]
  (get derivations key key))

(defn compute-derivations [blocks]
  (into {}
    (keep (fn [[key {:keys [type]}]]
            (when (nil? type)
              (throw (ex-info (str "The key " key " has no type")
                       {:key key})))
            (when (not (isa? key type))
              [key [type (if (vector? key)
                           (composite-keyword key)
                           key)]])))
    blocks))

(defn- init′
  ([{:keys [blocks components] :as config}]
   (init′ config (set (concat
                        (keys blocks)
                        (keys components)))))
  ([{:keys [name profiles blocks components]} keys]
   (let [derivations (compute-derivations blocks)
         ig-config   (into (or components {})
                       (map (fn [[key {:keys [config]}]]
                              [(derivations key) config]))
                       blocks)
         ig-config   (ig/expand ig-config)]
     (try
       {:system      (ig/init ig-config keys)
        :ig-config   ig-config
        :name        name
        :profiles    profiles
        :derivations derivations}
       (catch Exception e
         (halt! {:system (:system (ex-data e))})
         (throw e))))))

(defn init
  ([base+profiles]
   (init′ (apply merge-with-profiles base+profiles)))
  ([base+profiles keys]
   (init′ (apply merge-with-profiles base+profiles) keys)))

(defn ignite!
  ([base+profiles]
   (init base+profiles []))
  ([{:keys [system ig-config derivations] :as state} keys]
   (assoc state :system
                (ig/build
                  ig-config
                  (mapv (partial derivation derivations) keys)
                  (fn [k v]
                    (if (contains? system k)
                      (system k)
                      (ig/init-key k v)))
                  #'ig/wrapped-assert-key
                  ig/resolve-key))))

(defn quench! [{:keys [system ig-config] :as state} keys]
  (let [[halted-keys left-keys]
        (reduce
          (fn [[halted-keys left-keys] key]
            (when-some [value (system key)]
              (ig/halt-key! key value))
            [(conj halted-keys key) (disj left-keys key)])
          [#{} (set (clojure.core/keys system))]
          (#'ig/reverse-dependent-keys
            (#'ig/system-origin system)
            keys))]
    (vary-meta
      (assoc state :system
                   (ig/build ig-config left-keys
                     (fn [k _]
                       (system k))
                     #'ig/wrapped-assert-key
                     ig/resolve-key))
      assoc
      ::halted-keys halted-keys)))

(defn re-ignite! [state keys &
                  {:keys [with-descendents?]
                   :or   {with-descendents? true}}]
  (let [state (quench! state keys)]
    (ignite! state
      (if with-descendents?
        (-> state meta ::halted-keys)
        keys))))

(defn get-block [{:keys [system derivations]} key]
  (get system (derivation derivations key)))

(defn resolve-block [{:keys [derivations] :as instance} key]
  (ig/resolve-key (derivation derivations key) (get-block instance key)))

;; with system-instance

(defmacro with-system
  [[var base+profiles] & body]
  `(let [~var (init ~base+profiles)]
     (try
       ~@body
       (finally
         (halt! ~var)))))
