(ns au.com.bitpattern.engine.core
  "Implementation of engines."
  (:require
   [clojure.core.async :refer [go go-loop >!! chan alts!]]))

(defn- make-process [f]
  "Execute `f' for `ticks' iterations, returning the result on an channel."
  (fn [ticks state result-ch suspend-ch]
    (trampoline (fn rec [ticks state]
                  (let [{:keys [next-state result]} (apply f state)]
                    (cond
                      result (>!! result-ch result)
                      (= ticks 1) (>!! suspend-ch next-state)
                      :default (fn [] (rec (dec ticks) next-state)))))
                ticks state)))

(defn make-engine [f complete expire]
  "Create an engine for the given function.

An engine is a function which given a quantity of work to do, and callbacks on
complete and expire, will try to make progress towards its goal.  If it
completes, it will call `complete' with its returned value, otherwise it will
call expire with an engine which can be used to continue the computation.

Computation of `f' will be performed in another thread.

The function `f' must meet the following criteria:

  - Has a zero argument form which initiates the computation,
  - Should always return a map,
  - The returned map map must contain the key :result with the final value
    once the computation is complete.
  - Should return a map with the key :next-state if the computation is not
    complete, whose value is an vector of the arguments to which the function
    should next be applied.

`complete' is a function taking one argument which will be invoked when the
computation is complete.  The argument is the result of the computation of
`f'.

`expire' is a function taking 1 argument, which is an engine which can be
called to continue the computation.  The engine takes the number of ticks to
execute for as its argument."
  (let [process (make-process f)]
    (let [result-ch (chan)
          suspend-ch (chan)]
      (letfn [(eng [next-state ticks]
                (go
                  (process ticks next-state result-ch suspend-ch))
                nil)]
        (go-loop []
          (let [[v p] (alts! [result-ch suspend-ch])]
            (if (= p result-ch)
              (complete v)
              (do
                (expire (partial eng v)))))
          (recur))
        (partial eng nil)))))
